#
# Dark Channel ChannelServer Signal Handler Library
#
# Copyright (C) 2015 by DataCore GmbH
#     Amir Guindehi <amir@datacore.ch>
#

package DarkChannel::Node::ChannelServer::SignalHandler;

use warnings;
use strict;

use Carp;
use Data::Dumper;
use Getopt::Long;
use POSIX qw(strftime);

use DarkChannel::Utils::Log;
use DarkChannel::Utils::SessionStorage;

use DarkChannel::Node::ChannelServer::Conf;

# Note: POE's default event loop uses select().
# See CPAN for more efficient POE::Loop classes.
#
# Parameters to use POE are not treated as normal imports.
# Rather, they're abbreviated modules to be included along with POE.
use POE;

use Exporter;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

our $VERSION = 0.10;
our @ISA = qw( Exporter );
our @EXPORT = qw( dc_poe_signalhandler_initialize
                  dc_poe_signalhandler_spawn );

our @EXPORT_OK = qw();

#
# spawn channel server signal handler session
#
sub dc_poe_signalhandler_spawn()
{
    my $listener_host = $CONF->{node}->{channelserver}->{_default}->{host};
    my $listener_port = $CONF->{node}->{channelserver}->{_default}->{port};
    my $alias_listener = 'ChannelServer-Listener-' . $listener_host . ':' . $listener_port;
    my $alias_daemon = 'ChannelServer';

    my $debug = $CONF->{log}->{log_dbg_session_signal};

    #
    # client listener signal session
    #
    my $signal_session = POE::Session->create(

        options => { debug => $debug, trace => 0, default => 1 },

        inline_states => {
            _start => sub {
                my ($kernel, $heap, $session, $input) = @_[KERNEL, HEAP, SESSION, ARG0];

                # raport startup
                dc_log_dbg("channel server signal handler session created", 'SignalHandler');
                $kernel->alias_set($alias_daemon);

                # store alias
                $heap->{alias} = $alias_daemon;

                # register signal handlers
                $kernel->sig(INT => 'sig_INT');
                $kernel->sig(DIE => 'sig_DIE');
            },

            _stop => sub {
            },

            sig_DIE => sub {
                # $sig is 'DIE', $exception is the exception hash
                my($kernel, $heap, $session, $sig, $exception ) = @_[KERNEL, HEAP, SESSION, ARG0, ARG1];

                # show exception in log and
                my $msg = "died in $exception->{event}\n$exception->{error_str}";
                dc_log_err($msg, 'SignalHandler: SIGDIE: Exception: ChannelServer');

                # mark signal as handled
                $kernel->sig_handled();

                # request shutdown
                $kernel->yield('shutdown');
            },

            sig_INT => sub {
                # $sig is 'INT', $exception is the exception hash
                my($kernel, $heap, $session, $sig, $exception ) = @_[KERNEL, HEAP, SESSION, ARG0, ARG1];

                # show exception in log and
                dc_log_info("signal received!", 'SignalHandler: SIGINT');

                # mark signal as handled
                $kernel->sig_handled();

                # request shutdown
                $kernel->yield('shutdown');
            },

            shutdown => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];

                # kill sshtunnel
                DarkChannel::Node::ChannelServer::dc_channelserver_sshtunnel_shutdown();

                # send 'shutdown' event to all connected client sessions
                for my $sid (dc_session_all())
                {
                    my $alias = dc_session_data_get($sid, 'alias');

                    if ($alias) {
                        dc_log_dbg("sending shutdown event to '" . $alias . "'", 'SignalHandler');

                        # send 'shutdown' to client session
                        $kernel->post($alias, 'shutdown' );
                    }
                }

                # request delayed shutdown, so that all log messages may still pass
                $kernel->delay('shutdown_listener' => 1);
            },

            shutdown_listener => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];

                # send 'shutdown' event to listener session
                dc_log_dbg("sending shutdown event to '" . $alias_listener . "'", 'SignalHandler');
                $kernel->post($alias_listener, 'shutdown');
            }

        },
    );
}

sub dc_poe_signalhandler_initialize()
{
    # initialize this module
    dc_log_dbg("initializing DarkChannel::Node::ChannelServer::SignalHandler");

    return 1;
}

1;
