#
# Dark Channel IRC Core Backend Client Library
#
# Copyright (C) 2015 by DataCore GmbH
#     Amir Guindehi <amir@datacore.ch>
#

package DarkChannel::Node::Client::Core::IRC;
BEGIN
{
  $DarkChannel::Node::Client::Core::IRC::VERSION = '1.54';
}

use strict;
use warnings;
use Carp qw(croak confess);

use Data::Dumper;
use POSIX 'strftime';
use List::Util qw(sum);
use MIME::Base64;
use IRC::Utils qw(uc_irc parse_mode_line unparse_mode_line normalize_mask
                  matches_mask gen_mode_change is_valid_nick_name
                  is_valid_chan_name);

use DarkChannel::Utils::Log;
use DarkChannel::Utils::SessionStorage;

use DarkChannel::Node::Client::Core::Log;

use DarkChannel::Node::Client::Conf;

use POE;
use POE::Component::Server::IRC::Common qw(chkpasswd);
use POE::Component::Server::IRC::Plugin qw(:ALL);

use base qw(POE::Component::Server::IRC::Backend);

sub spawn {
    my ($package, %args) = @_;
    $args{lc $_} = delete $args{$_} for keys %args;
    my $config = delete $args{config};
    my $debug = delete $args{debug};
    my $alias = delete $args{alias};
    my $alias_customer = delete $args{alias_customer};
    my $now = strftime "%Y-%m-%d-%H%M%S", localtime;
    my $self = $package->create(
        ($debug ? (raw_events => 1) : ()),
        %args,
        states => [
#            [qw(add_spoofed_nick del_spoofed_nick)],
            {
#                map { +"daemon_cmd_$_" => '_spoofed_command' }
#                    qw(join part mode kick topic nick privmsg notice gline
#                       kline unkline sjoin locops wallops operwall)
            },
        ],
    );

    # setup $self
    $self->{debug} = $debug;
    $self->setup_error_codes();
    $self->{alias} = $alias;
    $self->{alias_customer} = $alias_customer;

    # create Core IRC listener session
    my $sid = $self->session_id();
    my $sdata = {
        type => 'Core::IRC',
        alias_old => $self->session_alias(),
        alias => $alias,
        alias_customer => $alias_customer,
        time_create => $now,
        time_create_epoc => time(),
        clients => {},
    };
    dc_session_register($sid, $sdata);

    return $self;
}

sub setup_error_codes
{
    my $self = shift;

    $self->{Error_Codes} = {
        401 => [1, "No such nick/channel"],
        402 => [1, "No such server"],
        403 => [1, "No such channel"],
        404 => [1, "Cannot send to channel"],
        405 => [1, "You have joined too many channels"],
        406 => [1, "There was no such nickname"],
        407 => [1, "Too many targets"],
        408 => [1, "No such service"],
        409 => [1, "No origin specified"],
        411 => [0, "No recipient given (%s)"],
        412 => [0, "No text to send"],
        413 => [1, "No toplevel domain specified"],
        414 => [1, "Wildcard in toplevel domain"],
        415 => [1, "Bad server/host mask"],
        421 => [1, "Unknown command"],
        422 => [0, "MOTD File is missing"],
        423 => [1, "No administrative info available"],
        424 => [1, "File error doing % on %"],
        431 => [1, "No nickname given"],
        432 => [1, "Erroneous nickname"],
        433 => [1, "Nickname is already in use"],
        436 => [1, "Nickname collision KILL from %s\@%s"],
        437 => [1, "Nick/channel is temporarily unavailable"],
        441 => [1, "They aren\'t on that channel"],
        442 => [1, "You\'re not on that channel"],
        443 => [2, "is already on channel"],
        444 => [1, "User not logged in"],
        445 => [0, "SUMMON has been disabled"],
        446 => [0, "USERS has been disabled"],
        451 => [0, "You have not registered"],
        461 => [1, "Not enough parameters"],
        462 => [0, "Unauthorised command (already registered)"],
        463 => [0, "Your host isn\'t among the privileged"],
        464 => [0, "Password mismatch"],
        465 => [0, "You are banned from this server"],
        466 => [0, "You will be banned from this server"],
        467 => [1, "Channel key already set"],
        471 => [1, "Cannot join channel (+l)"],
        472 => [1, "is unknown mode char to me for %s"],
        473 => [1, "Cannot join channel (+i)"],
        474 => [1, "Cannot join channel (+b)"],
        475 => [1, "Cannot join channel (+k)"],
        476 => [1, "Bad Channel Mask"],
        477 => [1, "Channel doesn\'t support modes"],
        478 => [2, "Channel list is full"],
        481 => [0, "Permission Denied- You\'re not an IRC operator"],
        482 => [1, "You\'re not channel operator"],
        483 => [0, "You can\'t kill a server!"],
        484 => [0, "Your connection is restricted!"],
        485 => [0, "You\'re not the original channel operator"],
        491 => [0, "No O-lines for your host"],
        501 => [0, "Unknown MODE flag"],
        502 => [0, "Cannot change mode for other users"],
    };
    return;
}

sub IRCD_plugin_add {
    my ($self, $plugin) = splice @_, 0, 2;
    my $sid = $self->session_id();
    my $alias = dc_session_data_get($sid, 'alias');

    # correct alias
    my $alias_irc = 'Core-Listener-IRC';
    my @aliases = $poe_kernel->alias_list();
    $poe_kernel->alias_set($alias // $alias_irc);
    foreach (@aliases) {
        $poe_kernel->alias_remove($_);
    }
    return PCSI_EAT_NONE;
}

sub IRCD_plugin_del {
    my ($self, $plugin) = splice @_, 0, 2;
    my $sid = $self->session_id();
    my $alias = dc_session_data_get($sid, 'alias');

    dc_log_dbg("ignoring event", $alias . ": Event: ircd_plugin_del");
    return PCSI_EAT_CLIENT;
}

sub IRCD_listener_add  {
    my ($self, $plugin) = splice @_, 0, 2;
    my $sid = $self->session_id();
    my $alias = dc_session_data_get($sid, 'alias');

    dc_log_dbg("ignoring event", $alias . ": Event: ircd_listener_add");
    return PCSI_EAT_CLIENT;
}


sub IRCD_connection {
    my ($self, $ircd) = splice @_, 0, 2;
    pop @_;
    my ($conn_id, $peeraddr, $peerport, $sockaddr, $sockport, $needs_auth) = map { ${ $_ } } @_;
    my $sid = $self->session_id();
    my $alias = dc_session_data_get($sid, 'alias');
    my $now = strftime "%Y-%m-%d-%H%M%S", localtime;
    my $sap_peer = $peeraddr . ':' . $peerport;
    my $sap_sock = $sockaddr . ':' . $sockport;
    my $sauth = {
        hostname => '',
        ident => '',
    };

    # remove stale connection session
    dc_session_data_delete($sid, 'clients', $conn_id);

    # setup new connection session
    dc_session_data_set($sid, 'clients', $conn_id, 'sap_peer', $sap_peer);
    dc_session_data_set($sid, 'clients', $conn_id, 'sap', $sap_sock);
    dc_session_data_set($sid, 'clients', $conn_id, 'seen', $now);
    dc_session_data_set($sid, 'clients', $conn_id, 'state_ircd', 'auth_needed');
    dc_session_data_set($sid, 'clients', $conn_id, 'state_ircd_cap', 'disabled');

    core_log_dbg("new connection (sid=" . $sid . ", conn_id=" . $conn_id . ", sap_sock=" . $sap_sock
               . ", sap_peer=" . $sap_peer . ", needs_auth=" . $needs_auth .")", $alias . ': IRCD_connection()');

    $self->send_event('core_client_connected', $sid, $conn_id);
    return PCSI_EAT_CLIENT;
}

sub IRCD_auth_done {
    my ($self, $ircd) = splice @_, 0, 2;
    pop @_;
    my ($conn_id, $ref) = map { ${ $_ } } @_;
    my $sid = $self->session_id();
    return PCSI_EAT_CLIENT unless dc_session_data_get($sid, 'clients', $conn_id);
    my $alias = dc_session_data_get($sid, 'alias');

    dc_session_data_set($sid, 'clients', $conn_id, 'user_auth', $ref);
    $self->_auth_done($sid, $conn_id);

    return PCSI_EAT_CLIENT;
}

sub IRCD_disconnected {
    my ($self, $ircd) = splice @_, 0, 2;
    pop @_;
    my ($conn_id, $errstr) = map { ${ $_ } } @_;
    my $sid = $self->session_id();
    return PCSI_EAT_CLIENT unless dc_session_data_get($sid, 'clients', $conn_id);
    my $alias = dc_session_data_get($sid, 'alias');

    # remove connection session
    dc_session_data_delete($sid, 'clients', $conn_id);

    core_log_dbg("disconnected (conn_id=" . $conn_id .")", $alias . ': IRCD_disconnected()');
    return PCSI_EAT_CLIENT;
}

sub IRCD_raw_input {
    my ($self, $ircd) = splice @_, 0, 2;
    return PCSI_EAT_CLIENT if !$self->{debug};
    my $sid = $self->session_id();
    my $alias = dc_session_data_get($sid, 'alias');
    my $conn_id = ${ $_[0] };
    my $input   = ${ $_[1] };
    core_log_dbg("<<< $conn_id: $input", $alias . ': RAW');
    return PCSI_EAT_CLIENT;
}

sub IRCD_raw_output {
    my ($self, $ircd) = splice @_, 0, 2;
    return PCSI_EAT_CLIENT if !$self->{debug};
    my $sid = $self->session_id();
    my $alias = dc_session_data_get($sid, 'alias');
    my $conn_id = ${ $_[0] };
    my $output  = ${ $_[1] };
    core_log_dbg(">>> $conn_id: $output", $alias . ': RAW');
    return PCSI_EAT_CLIENT;
}

sub _cmd_allowed {
    my ($self, $sid, $conn_id, $event) = splice @_, 0, 4;
    my $alias = dc_session_data_get($sid, 'alias');
    my $state = dc_session_data_get($sid, 'clients', $conn_id, 'state_ircd') // return 0;
    my $cmd = 'unknown';
    my $err = '';

    if ($event =~ /CMD_(.+)/) {
        $cmd = uc($1);
    }
    unless($cmd) {
        core_log_dbg("event '" . $event . "' in state '" . $state . "' not allowed", $alias . ": _cmd_allowed()");
        return (0, "event '" . $event . "' in state '" . $state . "' not allowed");
    }

    # allow NOTICE in every case
    return (1, '') if ($cmd eq 'NOTICE');

    my @cmd_always    = qw( MOTD PING QUIT );
    my @cmd_pre_auth  = qw( NICK USER PASS USERHOST CAP AUTHENTICATE );
    my @cmd_key_gen   = qw( USERHOST );
    my @cmd_connected = qw( JOIN PART PRIVMSG NAMES );

    my @allowed = @cmd_always;

    if($state eq 'auth_needed') {
        push(@allowed, @cmd_pre_auth);
        $err = "Please authenticate your nick using the '/pass <pw>' command first!";
    }
    elsif($state eq 'auth_received') {
        $err = "Please wait until your authentication is processed!";
    }
    elsif($state eq 'key_material_generation') {
        push(@allowed, @cmd_key_gen);
        $err = "Please wait until your key material has been generated!";
    }
    elsif($state eq 'key_material_found') {
        push(@allowed, @cmd_key_gen);
        $err = "Please wait until the key material has been loaded!";
    }
    elsif($state eq 'key_material_failed') {
        $err = "Key material generation has failed. Please contact an administrator!";
    }
    elsif($state eq 'connecting') {
        $err = "Please wait until your session is connected to the channel server!";
    }
    elsif($state eq 'connected') {
        push(@allowed, @cmd_connected);
        $err = "Please use one of the following commands: " . join(', ', @allowed);
    }
    elsif($state eq 'disconnected') {
        $err = "Your channel server has disconnected. Please reconnect!";
    }

    # return success if command allowed
    if (grep(/^$cmd$/, @allowed)) {
        #core_log_dbg("command '" . $cmd . "' in state '" . $state . "' allowed", $alias . ": _cmd_allowed()");
        return (1, '')
    }

    # return failure & error message
    core_log_dbg("command '" . $cmd . "' in state '" . $state . "' not allowed: " . $err, $alias . ": _cmd_allowed()");
    return (0, $err);
}

sub send_output {
    my ($self, $cmds, $conn_id, $sid) = @_;

    # allow single outputs without array of hashes: make arrayref if hashref
    my $output = (ref($cmds) eq 'HASH') ? [ $cmds ] : $cmds;

    # handle array of hashes as output
    if (ref($output) eq 'ARRAY') {
        foreach my $cmd (@{ $output }) {
            # check if reply needed
            if (ref($cmd) eq 'HASH') {
                # send all answer commands
                $self->SUPER::send_output($cmd, $conn_id);
            }
            # check if error reply needed
            if(ref($cmd) eq 'ARRAY') {
                # send error
                my $err = shift(@{ $cmd });
                my $server = $CONF->{node}->{core}->{ircd}->{_default}->{server_name} // 'unknown.universe';
                my $nick = dc_session_data_get($sid, 'clients', $conn_id, 'user_nick');

                if (defined $self->{Error_Codes}{$err}) {
                    my $output = {
                        command => $err,
                        prefix  => $server,
                        params  => [$nick],
                    };
                    if ($self->{Error_Codes}{$err}[0] > 0) {
                        for (my $i = 1; $i <= $self->{Error_Codes}{$err}[0]; $i++) {
                            push(@{ $output->{params} }, shift(@{ $cmd }));
                        }
                    }
                    if ($self->{Error_Codes}{$err}[1] =~ /%/) {
                        push @{ $output->{params} }, sprintf($self->{Error_Codes}{$err}[1], @{ $cmd });
                    }
                    else {
                        push @{ $output->{params} }, $self->{Error_Codes}{$err}[1];
                    }
                    $self->SUPER::send_output($output, $conn_id);
                }
            }

        }
    }
}

sub _default {
    my ($self, $ircd, $event) = splice @_, 0, 3;
    my $sid = $self->session_id();
    my $alias = dc_session_data_get($sid, 'alias');

    return PCSI_EAT_NONE if ($event =~ /^IRCD_core_/);
    if ($event !~ /^IRCD_cmd_/) {
        dc_log_dbg("event not of type 'IRCD_cmd_*', ignoring" , $alias . ': Event: ' . $event);
        return PCSI_EAT_NONE
    }
    pop @_;
    my ($conn_id, $input) = map { $$_ } @_;
    return PCSI_EAT_CLIENT unless dc_session_data_get($sid, 'clients', $conn_id);

    # mark client seen
    my $now = strftime "%Y-%m-%d-%H%M%S", localtime;
    dc_session_data_set($sid, 'clients', $conn_id, 'seen', $now);

    # check command function
    $event =~ s/IRCD_cmd_/CMD_/g;
    if ($self->can($event)) {
        my $debug_event = ($CONF->{log}->{log_dbg_session_core_ircd_event}) // 0;
        my $debug_output = ($CONF->{log}->{log_dbg_session_core_ircd_output}) // 0;

        # log the event
        core_log_dbg(Dumper($input), $alias . ': Event: ' . $event . '[' . $conn_id . ']') if ($debug_event);

        my ($allowed, $err) = $self->_cmd_allowed($sid, $conn_id, $event);
        if ($allowed) {
            # call command function
            my $cret = $self->$event($sid, $conn_id, $input->{params}, $input);

            # send returned commands
            core_log_dbg(Dumper($cret), $alias . ': Output: ' . $event . '[' . $conn_id . ']') if ($debug_output);
            $self->send_output($cret, $conn_id, $sid);
        }
        # command not allowed
        else {
            # send error 481
            #my $server = $CONF->{node}->{core}->{ircd}->{_default}->{server_name};
            #my $nick = dc_session_data_get($sid, 'clients', $conn_id, 'user_nick');
            #my $output = {
            #    command => '481',
            #    prefix  => $server,
            #    params  => [ $nick, "Unauthorized command!" ],
            #};
            #$self->send_output($output, $conn_id, $sid);

            # inform client: error
            $self->send_notice($sid, $conn_id, $err) if ($err);
            dc_log_dbg("ignoring not allowed event:\n" . Dumper($input),$alias . ': Event: ' . $event . '[' . $conn_id . ']');
        }
    }
    else {
        # log otherwise
        dc_log_dbg("ignoring not implemented event:\n" . Dumper($input),$alias . ': Event: ' . $event . '[' . $conn_id . ']');
    }

    return PCSI_EAT_CLIENT;
}

sub _client_register {
    my ($self, $sid, $conn_id) = @_;
    my $alias = dc_session_data_get($sid, 'alias');
    my $server = $CONF->{node}->{core}->{ircd}->{_default}->{server_name} // 'unknown.universe';
    my $network = $CONF->{node}->{core}->{ircd}->{_default}->{server_network} // 'Unknown.NET';
    my $nick = dc_session_data_get($sid, 'clients', $conn_id, 'user_nick') // '';
    my $version = '1.00';
    my $server_created = strftime("This server was created %a %h %d %Y at %H:%M:%S %Z",
                                  localtime(dc_session_data_get($sid, 'time_create_epoc')));

    my $ref = [];

    core_log_dbg("server '" . $server . "', network '" . $network . "'", $alias . ': Command: WELCOME [' . $conn_id . ']');
    push @$ref, {
        prefix  => $server,
        command => '001',
        params  => [ $nick, "Welcome to the $network DarkChannel Relay Chat network $nick" ],
    };
    push @$ref, {
        prefix  => $server,
        command => '002',
        params  => [ $nick, "Your host is $server, running version $version" ],
    };
    push @$ref, {
        prefix  => $server,
        command => '003',
        params  => [ $nick, $server_created ],
    };
    push @$ref, {
        prefix   => $server,
        command  => '004',
        colonify => 0,
        params   => [ $nick, $server, $version, 'Dilowz', 'biklmnopstveIh', 'bkloveIh' ],
    };

    # send output
    #dc_log_dbg(Dumper($ref), $alias . ': Output: _client_register [' . $conn_id . ']');
    $self->send_output($_, $conn_id, $sid) for (@$ref);
    return;
}

sub _auth_done {
    my ($self, $sid, $conn_id) = @_;
    my $alias = dc_session_data_get($sid, 'alias');
    my $auth = dc_session_data_get($sid, 'clients', $conn_id, 'user_auth') // '';
    my $nick = dc_session_data_get($sid, 'clients', $conn_id, 'user_nick') // '';
    my $user = dc_session_data_get($sid, 'clients', $conn_id, 'user_name') // '';
    my $realname = dc_session_data_get($sid, 'clients', $conn_id, 'user_realname') // '';
    my $pass = dc_session_data_get($sid, 'clients', $conn_id, 'user_pass') // '';
    my $cap = dc_session_data_get($sid, 'clients', $conn_id, 'state_ircd_cap') // 'disabled';

    my $debug = ($CONF->{log}->{log_dbg_session_core_ircd_auth}) // 0;
    core_log_dbg("auth done (conn_id=" . $conn_id .", auth_done=" . ($auth ? '1':'0') . ", nick=" . $nick
               . ", pass=***)",$alias . ': _auth_done()') if ($debug);

    if (($cap eq 'disabled') || ($cap eq 'cap_sasl_success')) {
        if ($auth && $nick && $realname && $pass) {
            # remove user_pass from session state and pass it to core
            dc_session_data_delete($sid, 'clients', $conn_id, 'user_pass');
            # send event 'core_client_auth_done'
            dc_session_data_set($sid, 'clients', $conn_id, 'state_ircd', 'auth_received');
            $self->send_event('core_client_auth_done', $sid, $conn_id, $nick, $pass);
            return;
        }
    }
    if (($cap eq 'disabled') || ($cap eq 'cap_auth_success')) {
        if ($auth && $nick && $user) {
            # for nick name change if illegal nickname used
            if ($nick =~ /^0x/) {
                my $nick_new = 'guest' . $conn_id;
                # inform client on forced nick change
                $self->send_notice($sid, $conn_id, 'Invalid nick ' . $nick);
                $self->send_notice($sid, $conn_id, 'Forced nick change to ' . $nick_new);
                # send NICK command to client
                $self->send_nick($sid, $conn_id, $nick_new);
            }

            # send initial server responses 001..005
            $self->_client_register($sid, $conn_id);
            # send MOTD
            $self->send_event('cmd_motd', $conn_id, { command => 'MOTD' });
            # send event 'core_client_identified' after MOTD
            $self->send_event('core_client_identified', $sid, $conn_id, $nick);
        }
    }

    return;
}

sub _user_host {
    my ($self, $sid, $conn_id, $nick, $user, $hostname) = @_;
    confess('no $sid or $conn_id') unless ($sid && $conn_id);

    $nick = dc_session_data_get($sid,'clients',  $conn_id, 'user_nick') unless ($nick);
    $user = dc_session_data_get($sid,'clients',  $conn_id, 'user_name') unless ($user);
    $hostname = $CONF->{node}->{core}->{ircd}->{_default}->{server_name} unless ($hostname);

    return ($nick // 'unknown') . '@' . $hostname;
}

sub _user_full {
    my ($self, $sid, $conn_id, $nick, $user, $hostname) = @_;
    confess('no $sid or $conn_id') unless ($sid && $conn_id);

    $nick = dc_session_data_get($sid,'clients',  $conn_id, 'user_nick') unless ($nick);
    #$user = dc_session_data_get($sid,'clients',  $conn_id, 'user_name') unless ($user);
    #$hostname = $CONF->{node}->{core}->{ircd}->{_default}->{server_name} unless ($hostname);
    #return $nick . '!' . $user . '@' . $hostname;

    return $nick . '!' . $self->_user_host($sid, $conn_id, $nick, $user, $hostname);
}

# XXX: TODO: fix similar to send_privmsg
sub send_notice
{
    my ($self, $sid, $conn_id, $msg, $prefix) = @_;

    confess('no $sid or $conn_id or $msg') unless ($sid && $conn_id && $msg);

    my $prefix_default = $CONF->{node}->{core}->{ircd}->{_default}->{server_prefix};
    $prefix = $prefix_default unless(defined($prefix));
    $prefix .= ' ' if ($prefix);

    # send NOTICE command to client
    my $server = $CONF->{node}->{core}->{ircd}->{_default}->{server_name};
    my $nick = dc_session_data_get($sid, 'clients', $conn_id, 'user_nick');
    my @lines = split(/\n/, $msg);
    #$self->send_event('cmd_notice', $conn_id, { command => 'NOTICE', params => [ $prefix . $_ ]}) for (@lines);
    $self->send_output({ command => 'NOTICE', params  => [$nick, $prefix . $_] }, $conn_id, $sid)
        for (@lines);
    return;
}

sub send_privmsg
{
    my ($self, $sid, $conn_id, $recipient, $key_id, $msg, $prefix) = @_;
    confess('no $sid or $conn_id or $msg') unless ($sid && $conn_id && $msg);

    $prefix = '' unless(defined($prefix));
    $prefix .= ' ' if ($prefix);

    # send PRIVMSG command to client
    my $nick = $key_id ? $key_id : dc_session_data_get($sid, 'clients', $conn_id, 'user_nick');
    my $nick_full = $self->_user_full($sid, $conn_id, $key_id);
    my $alias = dc_session_data_get($sid, 'alias');
    my @lines = split(/\n/, $msg);

    core_log_dbg("sending PRIVMSG '" . $recipient . "' <" . $nick_full . "> to client", $alias . ' [' . $conn_id . ']');
    for (@lines) {
        my $text = $prefix . $_;
        $self->send_output({ prefix  => $nick_full, command => 'PRIVMSG', params  => [$recipient, $text] }, $conn_id, $sid);
    }
    return;
}

sub send_join
{
    my ($self, $sid, $conn_id, $channel, $key_id) = @_;
    confess('no $sid or $conn_id or $channel') unless ($sid && $conn_id && $channel);

    # send JOIN command to client
    my $nick = $key_id ? $key_id : dc_session_data_get($sid, 'clients', $conn_id, 'user_nick');
    my $nick_full = $self->_user_full($sid, $conn_id, $key_id);
    my $alias = dc_session_data_get($sid, 'alias');

    core_log_dbg("sending JOIN '" . $channel . "' <" . $nick_full . "> to client", $alias . ' [' . $conn_id . ']');
    $self->send_output({ prefix  => $nick_full, command => 'JOIN', params  => [$channel] }, $conn_id, $sid);
    return;
}

sub send_part
{
    my ($self, $sid, $conn_id, $channel, $key_id, $msg) = @_;
    confess('no $sid or $conn_id or $channel') unless ($sid && $conn_id && $channel);

    # send PART command to client
    my $nick = $key_id ? $key_id : dc_session_data_get($sid, 'clients', $conn_id, 'user_nick');
    my $nick_full = $self->_user_full($sid, $conn_id, $key_id);
    my $alias = dc_session_data_get($sid, 'alias');
    $msg = $nick . ' has left ' . $channel unless($msg);

    core_log_dbg("sending PART '" . $channel . "' <" . $nick_full . "> to client: " . $msg, $alias . ' [' . $conn_id . ']');
    $self->send_output({ prefix  => $nick_full, command => 'PART', params  => [$channel, $msg] }, $conn_id, $sid);
    return;
}

sub send_nick
{
    my ($self, $sid, $conn_id, $nick) = @_;
    confess('no $sid or $conn_id or $nick') unless ($sid && $conn_id && $nick);

    # send NICK command to client
    my $alias = dc_session_data_get($sid, 'alias');
    my $nick_full = $self->_user_full($sid, $conn_id);

    # change stored nick name to new nick
    dc_session_data_set($sid,'clients',  $conn_id, 'user_nick', $nick);

    core_log_dbg("sending NICK '" . $nick . "' to client", $alias . ' [' . $conn_id . ']');
    $self->send_output({ prefix  => $nick_full, command => 'NICK', params  => [$nick] }, $conn_id, $sid);
    return;
}

sub send_rpl_notopic
{
    my ($self, $sid, $conn_id, $channel) = @_;
    confess('no $sid or $conn_id or $channel') unless ($sid && $conn_id && $channel);

    # send RPL_NOTOPIC (331) command to client
    my $server = $CONF->{node}->{core}->{ircd}->{_default}->{server_name};
    my $alias = dc_session_data_get($sid, 'alias');
    my $nick = dc_session_data_get($sid, 'clients', $conn_id, 'user_nick');
    my $msg = 'No topic is set';

    core_log_dbg("sending RPL_NOTOPIC (331) '" . $channel . "' to client: " . $msg, $alias . ' [' . $conn_id . ']');
    $self->send_output({ prefix => $server, command => '331', params  => [$nick, $channel, $msg] }, $conn_id, $sid);
    return;
}

sub send_rpl_topic
{
    my ($self, $sid, $conn_id, $channel, $topic) = @_;
    confess('no $sid, $conn_id, $channel or $topic') unless ($sid && $conn_id && $channel && $topic);

    # send RPL_NOTOPIC (332) command to client
    my $server = $CONF->{node}->{core}->{ircd}->{_default}->{server_name};
    my $alias = dc_session_data_get($sid, 'alias');
    my $nick = dc_session_data_get($sid, 'clients', $conn_id, 'user_nick');

    core_log_dbg("sending RPL_TOPIC (332) '" . $channel . "' to client: " . $topic, $alias . ' [' . $conn_id . ']');
    $self->send_output({ prefix => $server, command => '332', params  => [$nick, $channel, $topic] }, $conn_id, $sid);
    return;
}

#
# $names is a hash of channel keys pointing to channel member name arrays
#
sub send_rpl_names
{
    my ($self, $sid, $conn_id, $names) = @_;
    confess('no $sid or $conn_id or $channel') unless ($sid && $conn_id && $names);

    # send reply to NAMES query
    my $server = $CONF->{node}->{core}->{ircd}->{_default}->{server_name};
    my $alias = dc_session_data_get($sid, 'alias');
    my $nick = dc_session_data_get($sid, 'clients', $conn_id, 'user_nick');
    my $msg_end = 'End of NAMES list';
    my $ref = [];

    foreach my $channel (keys %{ $names }) {
        my $members = join(' ', @{ $names->{$channel} });
        my $params = [ $nick, '=', $channel, $members ];

        # XXX: TODO: split into multiple RPL_NAMREPLY if too many

        # send RPL_NAMREPLY (353) command to client
        push(@{ $ref }, { prefix => $server, command => '353', params => $params });
        core_log_dbg("sending RPL_NAMREPLY (353) '" . $channel . "' to client: " . $members, $alias . ' [' . $conn_id . ']');

        # send RPL_ENDOFNAMES (366) command to client
        core_log_dbg("sending RPL_ENDOFNAMES (366) '" . $channel . "' to client: " . $members, $alias . ' [' . $conn_id . ']');
        push(@{ $ref }, { prefix => $server, command => '366', params => [$nick, $channel, $msg_end] });
    }

    $self->send_output($ref, $conn_id, $sid);
    return;
}

sub send_rpl_userhost
{
    my ($self, $sid, $conn_id, $nicknames) = @_;
    confess('no $sid or $conn_id or $nicknames') unless ($sid && $conn_id && $nicknames);

    # send reply to NAMES query
    my $server = $CONF->{node}->{core}->{ircd}->{_default}->{server_name};
    my $alias = dc_session_data_get($sid, 'alias');
    my $nick = dc_session_data_get($sid, 'clients', $conn_id, 'user_nick');
    my $reply = [];

    my $away_true = '+';
    my $away_false = '-';
    my $op_false = '';
    my $op_true = '*';

    # generate RPL_USERHOST arguments
    foreach my $nickname (@{ $nicknames }) {
        push(@{ $reply }, $nickname . $op_false . '=' . $away_false . $self->_user_host($sid, $conn_id));
    }
    my $params = [ $nick, @{ $reply } ];

    # send RPL_USERHOST (302) command to client
    core_log_dbg("sending RPL_USERHOST (302) for '" . join("', '", @{$nicknames}) . "' to client: " . join(', ', @{$reply}),
               $alias . ' [' . $conn_id . ']');
    $self->send_output({ prefix => $server, command => '302', params => $params }, $conn_id, $sid);
    return;
}


sub terminate_connection
{
    my ($self, $sid, $conn_id, $msg, $prefix) = @_;
    confess('no $sid or $conn_id or $msg') unless ($sid && $conn_id && $msg);
    confess("session $sid not found!") unless (dc_session_exists($sid));

    my $alias = dc_session_data_get($sid, 'alias');
    if (my $c = dc_session_data_get($sid, 'clients', $conn_id)) {
        my $sap = $c->{sap_peer};

        # terminate connection
        core_log_dbg("terminate_connection for client from " . $sap . " with '" . $msg . "'",
                   $alias . '[' . $conn_id . ']');

        # send error message
        $self->send_output({ command => 'ERROR', params  => ['Closing Link: ' . $sap . ' (' . $msg . ')']}, $conn_id, $sid);

        # remove connection session
        dc_session_data_delete($sid, 'clients', $conn_id);
    }

    # terminate connection, component will terminate the connection the next time that the wheel input is flushed
    $self->disconnect($conn_id, $msg);

    return 1;
}

sub CMD_nick {
    my ($self, $sid, $conn_id, $params) = @_;
    my $alias = dc_session_data_get($sid, 'alias');
    my $nick = $params->[0] // '';
    my $nick_old = dc_session_data_get($sid, 'clients', $conn_id, 'user_nick');

    core_log_dbg("user nick '" . $nick . "'", $alias . ': Command: NICK [' . $conn_id . ']');

    # after the first NICK command: nick changed
    if ($nick_old) {
        # send NICK command to client
        $self->send_nick($sid, $conn_id, $nick);

        # send event 'core_client_identified'
        $self->send_event('core_client_identified', $sid, $conn_id, $nick)
    }
    # on the first NICK command
    else {
        # set user_nick in session
        dc_session_data_set($sid, 'clients', $conn_id, 'user_nick', $nick);

        # check if auth is finished
        $self->_auth_done($sid, $conn_id);
    }
    return PCSI_EAT_CLIENT;
}

sub CMD_user {
    my ($self, $sid, $conn_id, $params) = @_;
    my $alias = dc_session_data_get($sid, 'alias');
    my $username = $params->[0];
    my $usermode = $params->[1];
    my $realname = $params->[3];
    my $username_old = dc_session_data_get($sid, 'clients', $conn_id, 'user_name');

    core_log_dbg("user name '" . $username . "', real name '" . $realname . "'", $alias . ': Command: USER [' . $conn_id . ']');
    dc_session_data_set($sid, 'clients', $conn_id, 'user_name', $username);
    dc_session_data_set($sid, 'clients', $conn_id, 'user_realname', $realname);

    # check if auth is finished
    $self->_auth_done($sid, $conn_id) unless($username_old);
    return PCSI_EAT_CLIENT;
}

sub CMD_pass {
    my ($self, $sid, $conn_id, $params) = @_;
    my $alias = dc_session_data_get($sid, 'alias');
    my $pass = $params->[0] // '';

    core_log_dbg("user pass ***", $alias . ': Command: PASS [' . $conn_id . ']');
    dc_session_data_set($sid, 'clients', $conn_id, 'user_pass', $pass);

    # check if auth is finished
    $self->_auth_done($sid, $conn_id);
    return PCSI_EAT_CLIENT;
}

sub CMD_motd {
    my ($self, $sid, $conn_id, $params) = @_;
    my $alias = dc_session_data_get($sid, 'alias');
    my $ref = [];
    my $server = $CONF->{node}->{core}->{ircd}->{_default}->{server_name};
    my @motd = split(/\|/, $CONF->{node}->{core}->{ircd}->{_default}->{server_motd});
    my $nick = dc_session_data_get($sid, 'clients', $conn_id, 'user_nick');

    core_log_dbg("server '" . $server . "'", $alias . ': Command: MOTD [' . $conn_id . ']');
    if (@motd) {
        push @$ref, {
            prefix  => $server,
            command => '375',
            params  => [$nick, "- $server Message of the day - "],
        };
        push @$ref, {
            prefix  => $server,
            command => '372',
            params  => [$nick, "- $_"]
        } for @motd;
        push @$ref, {
            prefix  => $server,
            command => '376',
            params  => [$nick, "End of MOTD command"],
        };
    }
    else {
        # return err 422
        push(@$ref, ['422']);
    }

    return $ref;
}

sub CMD_join {
    my ($self, $sid, $conn_id, $params) = @_;
    my $alias = dc_session_data_get($sid, 'alias');
    my $channel = $params->[0] // '';
    my $ref = [];

    # send JOIN event to frontend
    core_log_dbg("channel '" . $channel . "'", $alias . ': Command: JOIN [' . $conn_id . ']');
    $self->send_event('core_client_cmd_join', $sid, $conn_id, $channel) if ($channel =~ /^(#[a-zA-Z0-9-]+)$/);

    return $ref;
}

sub CMD_part {
    my ($self, $sid, $conn_id, $params) = @_;
    my $alias = dc_session_data_get($sid, 'alias');
    my $channel = $params->[0] // '';
    my $ref = [];

    # send PART event to frontend
    core_log_dbg("channel '" . $channel . "'", $alias . ': Command: PART [' . $conn_id . ']');
    $self->send_event('core_client_cmd_part', $sid, $conn_id, $channel) if ($channel =~ /^(#[a-zA-Z0-9-]+)$/);

    return $ref;
}

sub CMD_privmsg {
    my ($self, $sid, $conn_id, $params) = @_;
    my $alias = dc_session_data_get($sid, 'alias');
    my $channel = $params->[0] // '';
    my $msg = $params->[1] // '';

    # send PRIVMSG event to frontend
    core_log_dbg("channel '" . $channel . "': " . $msg, $alias . ': Command: PRIVMSG [' . $conn_id . ']');
    $self->send_event('core_client_cmd_privmsg', $sid, $conn_id, $channel, $msg)
        if (($channel =~ /^(#[a-zA-Z0-9-]+)$/) && $msg);

    return PCSI_EAT_CLIENT;
}

#sub CMD_notice {
#    my ($self, $sid, $conn_id, $params) = @_;
#    my $alias = dc_session_data_get($sid, 'alias');
#    my $nick = dc_session_data_get($sid, 'clients', $conn_id, 'user_nick');
#    my $ref = [];
#    my $server = $CONF->{node}->{core}->{ircd}->{_default}->{server_name};
#    my $msg = $params->[0] // '';
#
#    core_log_dbg("server '" . $server . "'", $alias . ': Command: NOTICE [' . $conn_id . ']');
#    push @$ref, {
#        prefix  => $server,
#        command => 'NOTICE',
#        params  => [$nick, $msg],
#    };
#
#    return $ref;
#}

sub CMD_names {
    my ($self, $sid, $conn_id, $params) = @_;
    my $alias = dc_session_data_get($sid, 'alias');
    my $ref = [];

    # send NAMES event to frontend
    core_log_dbg("channels '" . join("', '", @{$params}) . "'", $alias . ': Command: NAMES [' . $conn_id . ']');
    $self->send_event('core_client_cmd_names', $sid, $conn_id, $params) if (@{ $params });

    return $ref;
}

sub CMD_quit {
    my ($self, $sid, $conn_id, $params) = @_;
    my $alias = dc_session_data_get($sid, 'alias');
    my $ref = [];

    # send QUIT event to frontend
    core_log_dbg("received from client", $alias . ': Command: QUIT [' . $conn_id . ']');
    $self->send_event('core_client_cmd_quit', $sid, $conn_id, $params) if (@{ $params });

    return $ref;
}

sub CMD_userhost {
    my ($self, $sid, $conn_id, $params) = @_;
    my $alias = dc_session_data_get($sid, 'alias');
    my $ref = [];
    my $state = dc_session_data_get($sid, 'clients', $conn_id, 'state_ircd');

    core_log_dbg("nickname: '" . join("', '", @{$params}) . "'", $alias . ': Command: USERHOST [' . $conn_id . ']');

    # don't send a reply until we are fully connected
    return $ref if ($state ne 'connected');

    # send RPL_USERHOST
    $self->send_rpl_userhost($sid, $conn_id, $params);

    return $ref;
}

sub CMD_ping {
    my ($self, $sid, $conn_id, $params) = @_;
    my $server = $CONF->{node}->{core}->{ircd}->{_default}->{server_name};
    my $alias = dc_session_data_get($sid, 'alias');
    my $arg = $params->[0] // '';
    my $ref = [];

    # send PONG event to client
    core_log_dbg("'" . $arg . "'", $alias . ': Command: PING [' . $conn_id . ']');
    core_log_dbg("sending PONG '" . $arg . "' to client", $alias . ' [' . $conn_id . ']');
    $self->send_output({ prefix => $server, command => 'PONG', params  => [$server, $arg] }, $conn_id, $sid);

    return $ref;
}

sub CMD_cap {
    my ($self, $sid, $conn_id, $params) = @_;
    my $server = $CONF->{node}->{core}->{ircd}->{_default}->{server_name};
    my $alias = dc_session_data_get($sid, 'alias');
    my $nick = dc_session_data_get($sid, 'clients', $conn_id, 'user_nick') // '*';
    my $cmd = $params->[0] // '';
    my $arg = $params->[1] // '';
    my $extentions = 'sasl';
    my $ref = [];

    core_log_dbg("received command from client: cmd=" . $cmd . ", arg=" . $arg,
                 $alias . ': Command: CAP [' . $conn_id . ']');

    # switch CAP command
    if ($cmd eq 'LS') {
        # CAP LS received, send supported CAP extentions
        my $args = [ $nick, 'LS', $extentions ];

        core_log_dbg("sending CAP LS (extentions '" . $extentions . "') to client", $alias . ' [' . $conn_id . ']');
        $self->send_output({ prefix => $server, command => 'CAP', params  => $args }, $conn_id, $sid);

        # create CAP state (guarantees that client registration gets delayed until CAP END)
        dc_session_data_set($sid, 'clients', $conn_id, 'state_ircd_cap', 'cap_ls_received');
    }
    elsif ($cmd eq 'END') {
        core_log_dbg("CAP END received ending authentication", $alias . ' [' . $conn_id . ']');

        # set CAP end state
        dc_session_data_set($sid, 'clients', $conn_id, 'state_ircd_cap', 'cap_sasl_success');

        # auth done
        $self->_auth_done($sid, $conn_id);
    }
    elsif ($cmd eq 'REQ') {
        if ($arg =~ /^$extentions$/) {
            # CAP ACK
            my $args = [ $nick, 'ACK', $extentions ];

            core_log_dbg("sending CAP ACK (extention '" . $extentions . "') for user '" . $nick . "' to client",
                         $alias . ' [' . $conn_id . ']');
            $self->send_output({ prefix => $server, command => 'CAP', params  => $args }, $conn_id, $sid);
            dc_session_data_set($sid, 'clients', $conn_id, 'state_ircd_cap', 'cap_req_ack_sent');
        }
        else {
            # CAP NAK
            my $args = [ $nick, 'NAK', $extentions ];

            core_log_dbg("sending CAP NAK (extention '" . $extentions . "') for user '" . $nick . "' to client",
                         $alias . ' [' . $conn_id . ']');
            $self->send_output({ prefix => $server, command => 'CAP', params  => $args }, $conn_id, $sid);
            dc_session_data_set($sid, 'clients', $conn_id, 'state_ircd_cap', 'cap_req_nak_sent');
        }
    }
    else {
        core_log_dbg("not supported command from client: cmd=" . $cmd . ", arg=" . $arg ,
                     $alias . ': Command: CAP [' . $conn_id . ']');
    }

    return $ref;
}

sub CMD_authenticate {
    my ($self, $sid, $conn_id, $params) = @_;
    my $server = $CONF->{node}->{core}->{ircd}->{_default}->{server_name};
    my $alias = dc_session_data_get($sid, 'alias');
    my $nick = dc_session_data_get($sid, 'clients', $conn_id, 'user_nick') // '*';
    my $cmd = $params->[0] // '';
    my $arg = $params->[1] // '';
    my $state_cap = dc_session_data_get($sid, 'clients', $conn_id, 'state_ircd_cap');
    my $mechanisms = 'PLAIN';
    my $ref = [];

    core_log_dbg("received command from client: cmd=" . $cmd . ", arg=" . $arg . ", state_cap=" . $state_cap,
                 $alias . ': Command: AUTHENTICATE [' . $conn_id . ']');

    # do noting if not in state CAP ACK sent
    return $ref if ($state_cap ne 'cap_req_ack_sent') && ($state_cap !~ /^cap_auth_/);

    # switch cap state
    if (($state_cap eq 'cap_req_ack_sent') || ($state_cap eq 'cap_auth_mech')) {
        # inital state, we accept PLAIN auth
        if ($cmd eq 'PLAIN') {
            # send AUTHENTICATE +
            my $args = [ '+' ];

            core_log_dbg("sending AUTHENTICATE + for user '" . $nick . "' to client",
                         $alias . ' [' . $conn_id . ']');
            $self->send_output({ command => 'AUTHENTICATE', params  => $args }, $conn_id, $sid);

            # update state
            dc_session_data_set($sid, 'clients', $conn_id, 'state_ircd_cap', 'cap_auth_plain_received');
        }
        # no PLAIN auth requested
        else {
            # send RPL_SASLMECHS (908) with supported AUTH mechanisms
            my $args = [ $nick, $mechanisms, 'are the available SASL mechanisms' ];

            core_log_dbg("sending RPL_SASLMECHS (908) (mechanisms '" . $mechanisms . "') for user '" . $nick . "' to client",
                         $alias . ' [' . $conn_id . ']');
            $self->send_output({ prefix => $server, command => '908', params  => $args }, $conn_id, $sid);

            # send ERR_SASLFAIL (904)
            $args = [ $nick, 'SASL authentication failed' ];

            core_log_dbg("sending ERR_SASLFAIL (904) for user '" . $nick . "' to client",
                         $alias . ' [' . $conn_id . ']');
            $self->send_output({ prefix => $server, command => '904', params  => $args }, $conn_id, $sid);

            # update state
            dc_session_data_set($sid, 'clients', $conn_id, 'state_ircd_cap', 'cap_auth_mech');
        }
    }
    elsif ($state_cap eq 'cap_auth_plain_received') {
        my $decoded = decode_base64($cmd);
        my ($nick_new, $user_new, $pass) = split(/\0/, $decoded);

        core_log_dbg("received AUTHENTICATION from client: auth=" . $cmd . " (nick=" . $nick_new . ", user=" . $user_new
                     . ", pass=***)", $alias . ' [' . $conn_id . ']');

        # send RPL_LOGGEDIN (900) with user accound
        my $nick_full = $self->_user_full($sid, $conn_id);
        my $args = [ $nick, $nick_full, $nick_new, 'You have authenticated as ' . $nick_new ];

        core_log_dbg("sending RPL_LOGGEDIN (900) (nick=" . $nick_new . ", user= " . $user_new . ", nick_full=" . $nick_full
                     . "') for user '" . $nick . "' to client", $alias . ' [' . $conn_id . ']');
        $self->send_output({ prefix => $server, command => '900', params  => $args }, $conn_id, $sid);

        # send RPL_SASLSUCCESS (903)
        $args = [ $nick, 'SASL authentication successful' ];

        core_log_dbg("sending RPL_SASLSUCCESS (903) for user '" . $nick_new . "' to client",
                     $alias . ' [' . $conn_id . ']');
        $self->send_output({ prefix => $server, command => '903', params  => $args }, $conn_id, $sid);

        # update state
        dc_session_data_set($sid, 'clients', $conn_id, 'state_ircd_cap', 'cap_auth_success');

        # store nick by changing nick if different (send NICK command to client)
        if ($nick ne $nick_new) {
            $self->send_notice($sid, $conn_id, 'Forced nick change to ' . $nick_new);
            $self->send_nick($sid, $conn_id, $nick_new);
        }

        # auth done
        $self->_auth_done($sid, $conn_id);

        # store passphrase
        dc_session_data_set($sid, 'clients', $conn_id, 'user_pass', $pass);
    }

    return $ref;
}

1;
