#
# Dark Channel Client Signal Handler Library
#
# Copyright (C) 2015 by DataCore GmbH
#     Amir Guindehi <amir@datacore.ch>
#

package DarkChannel::Node::Client::SignalHandler;

use warnings;
use strict;

use Carp;
use Data::Dumper;
use Getopt::Long;
use POSIX qw(strftime);

use DarkChannel::Utils::Log;
use DarkChannel::Utils::SessionStorage;

use DarkChannel::Node::Client::Conf;

# Note: POE's default event loop uses select().
# See CPAN for more efficient POE::Loop classes.
#
# Parameters to use POE are not treated as normal imports.
# Rather, they're abbreviated modules to be included along with POE.
use POE;

use Exporter;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

our $VERSION = 0.10;
our @ISA = qw( Exporter );
our @EXPORT = qw( dc_poe_signalhandler_initialize
                  dc_poe_signalhandler_spawn );

our @EXPORT_OK = qw();

#
# spawn channel server signal handler session
#
sub dc_poe_signalhandler_spawn()
{
    my $alias_term = 'Client-Terminal';
    my $alias_signalhandler = 'Client-SignalHandler';

    my $debug = $CONF->{log}->{log_dbg_session_signal};

    #
    # client listener signal session
    #
    my $signal_session = POE::Session->create(

        options => { debug => $debug, trace => 0, default => 1 },

        inline_states => {
            _start => sub {
                my ($kernel, $heap, $session, $input) = @_[KERNEL, HEAP, SESSION, ARG0];

                # raport startup
                dc_log_dbg("client signal handler session created", 'SignalHandler');
                $kernel->alias_set($alias_signalhandler);

                # store alias
                $heap->{alias} = $alias_signalhandler;

                # register signal handlers
                $kernel->sig(INT => 'sig_INT');
                $kernel->sig(DIE => 'sig_DIE');
                $kernel->sig(TSTP => 'sig_TSTP');
            },

            _stop => sub {
            },

            sig_DIE => sub {
                # $sig is 'DIE', $exception is the exception hash
                my($kernel, $heap, $session, $sig, $exception ) = @_[KERNEL, HEAP, SESSION, ARG0, ARG1];

                # show exception in log and
                my $msg = "died in $exception->{event}\n$exception->{error_str}";
                dc_log_err($msg, 'SIGDIE: Exception: Term');

                # mark exception as handled
                $poe_kernel->sig_handled();
            },

            sig_INT => sub {
                # $sig is 'INT', $exception is the exception hash
                my($kernel, $heap, $session, $sig, $exception ) = @_[KERNEL, HEAP, SESSION, ARG0, ARG1];

                # show exception in log and
                my $msg;
                $msg = "interrupted in $exception->{event}\n$exception->{error_str}" if ($exception);
                $msg = "interrupted (exception=undef)" unless($exception);
                dc_log_err($msg, 'SIGINT: Exception: Term');

                # mark exception as handled
                $poe_kernel->sig_handled();

                # request shutdown
                $kernel->yield('shutdown');
            },

            sig_TSTP => sub {
                # $sig is 'TSTP', $exception is the exception hash
                my($kernel, $heap, $session, $sig, $exception ) = @_[KERNEL, HEAP, SESSION, ARG0, ARG1];

                dc_log_dbg("suspending client application", 'SIGTSTP: Exception: Term');

                # pre suspend: leave curses and reset TTY to be usable by shell
                #$CURSES->leave_curses();
                $poe_kernel->call($alias_term, 'leave_curses');

                # reset TSTP signal handler to default
                local $SIG{TSTP} = 'DEFAULT';
                # kill this processes with signal TSTP and mark as handled when returning
                kill(TSTP => $$);
                $kernel->sig_handled();

                # post suspsend: reset Curses and redraw screen
                #$CURSES->reset_curses();
                $poe_kernel->call($alias_term, 'reset_curses');

                dc_log_dbg("resuming client application", 'Signal Handler: SIGTSTP: Exception');
            },

            shutdown => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];

                # close all channel server connections
                for my $sid (dc_session_all())
                {
                    # send 'shutdown' to channel server session if existing
                    if (my $alias = dc_session_data_get($sid, 'alias')) {
                        dc_log_dbg("sending shutdown event to '" . $alias . "'", 'Signal Handler');
                        $kernel->post($alias, 'shutdown' );
                    }
                }

                # request delayed shutdown, so that all log messages may still pass
                $kernel->delay('shutdown_term' => 1);
            },

            shutdown_term => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];

                # send 'shutdown' event to terminal session
                dc_log_dbg("sending shutdown event to '" . $alias_term . "'", 'SignalHandler');
                $kernel->post($alias_term, 'shutdown');
            },
        },
    );

    return 1;
}

sub dc_poe_signalhandler_initialize()
{
    # initialize this module
    dc_log_dbg("initializing DarkChannel::Node::Client::SignalHandler");

    return 1;
}

1;
