#
# Dark Channel Curses::UI:POE Terminal Library
#
# Copyright (C) 2015 by DataCore GmbH
#     Amir Guindehi <amir@datacore.ch>
#

package DarkChannel::Node::Client::Term;

use warnings;
use strict;

use Carp;
use Data::Dumper;
use POSIX qw(strftime);
use FileHandle;

use DarkChannel::Crypt::Base;

use DarkChannel::Utils::Log;
use DarkChannel::Utils::SessionStorage;

use DarkChannel::Proto::Client;

use DarkChannel::Node::Client::Term::UI;
use DarkChannel::Node::Client::Term::Command;
use DarkChannel::Node::Client::Term::History;

use DarkChannel::Node::Client::Conf;

# POE Debugging
#sub POE::Kernel::ASSERT_DEFAULT () { 1 }
#sub POE::Kernel::ASSERT_EVENTS  () { 1 }

# Note: POE's default event loop uses select().
# See CPAN for more efficient POE::Loop classes.
#
# Parameters to use POE are not treated as normal imports.
# Rather, they're abbreviated modules to be included along with POE.
use POE qw(Component::Client::TCP);

use Curses;
use Curses::UI::POE;

use Exporter;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

our $VERSION = 1.00;
our @ISA = qw( Exporter );
our @EXPORT_OK = qw();
our @EXPORT = qw( dc_poe_term_initialize
                  dc_poe_term_spawn
                  dc_poe_term_run
                  dc_poe_term_log_event );

# we moved the mainloop() function from Curses::UI::POE to this module
# this allowed us to fix a problem in the original function. see comments below
#
# note: we assume that there will never be two Curses::UI::POE sessions
my @modal_objects;
my @modal_callbacks;

# Curses object (POE)
my $CURSES = {};

# terminal client arguments
my $ARGS = {};

# global aliases
my $alias_terminal = 'Client-Terminal';
my $alias_signalhandler = 'Client-SignalHandler';

sub dc_poe_term_process_user_input()
{
    my $self = shift; # text_log
    my $input = $self->text();

    dc_log_dbg("user input: '" . $input . "'", 'Client-Terminal') if ($CONF->{log}->{log_dbg_session_terminal});

    if (($input =~ /^\/([^\s]+) (.*)$/) || ($input =~ /^\/([^\s]+)$/))
    {
        my ($cmd, $args) = ($1, $2 // '');

        # parse user commands which start with the '/' character
        dc_poe_term_cmd_process_usercmd($cmd, $args)
    }
    else
    {
        # process user input (no command)
        dc_poe_term_cmd_process_userinput($input)
    }

    # store command in history
    dc_poe_term_cmd_history_push($input) if ($input);

    # clear user input
    $self->text('');
}

sub dc_poe_term_log($$$)
{
    my $page = shift;
    my $msg = shift;
    my $prefix = shift // '';

    # append ': ' if not ending on '>'
    $prefix .= (not ($prefix =~ /\<.+\>/) ? ': ' : ' ') if ($prefix);

    my $now = strftime "%H:%M:%S", localtime;
    my @lines = split(/\n/, $msg);
    my $fullprefix = '[' . $now . '] ' . $prefix;
    my $lines = $fullprefix . join("\n" . $fullprefix, @lines);
    my $notebook = $CURSES->{ui}->{notebook};
    my $text_page = $CURSES->{ui}->{text}->{lc($page)};

    unless ($text_page) {
        my $keys = $CURSES->{ui}->{text} ? join(keys %{$CURSES->{ui}->{text}}) : '';
        dc_log_crit("dc_poe_term_log(): could not find page '" . lc($page) . "' received as '" . $page .
                    "' in CURSES->{ui}->{text} which contains currently '" . $keys . "'", 'Terminal Log');
        dc_log_crit('dc_poe_term_log(): ignoring message: ' . $lines);
        return;
    }

    my $text = $text_page->text();
    my $newtext = $text . "\n" . $lines;

    $text_page->text($newtext);
    $text_page->cursor_to_end();
    $text_page->parent()->parent()->draw();

}

sub dc_poe_term_spawn()
{
    my $mouse_support = $ARGS->{mouse_support};
    my $color_support = $ARGS->{color_support};
    my $irc_core_support = $ARGS->{irc_core_support};
    my $debug = $CONF->{log}->{log_dbg_session_curses};

    dc_log_dbg("creating client terminal statemachine (color_support=" . $ARGS->{color_support} .
               ", mouse_support=" . $ARGS->{mouse_support} . ")", $alias_terminal);

    # create Curses::UI POE session
    $CURSES = new Curses::UI::POE(
        -color_support => $ARGS->{color_support},
        -mouse_support => $ARGS->{mouse_support},
        -clear_on_exit => 1,
        -debug => $debug,

        inline_states => {
            _start => sub {
                my ($kernel, $heap, $session, $input ) = @_[KERNEL, HEAP, SESSION, ARG0];

                # register signal handlers
                #$kernel->sig(DIE => 'sig_DIE');
                #$kernel->sig(TSTP => 'sig_TSTP');

                # set session alias
                $kernel->alias_set($alias_terminal);

                # enable logging to terminal and replay prior messages
                dc_log_dbg("activating terminal logging and replaying prior log messages", $alias_terminal);
                dc_log_set_terminal($CURSES->{ui}->{text}->{log});
            },

            _stop => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
                $CURSES->leave_curses();
            },

            _default => sub {
                my ($session, $event, $args) = @_[SESSION, ARG0, ARG1];
                my $msg = "caught unhandled event $event (session=" . $session->ID . ") with (@$args)!";
                dc_log_crit($msg, $alias_terminal);
            },

            _child => sub {
                my($reason, $child ) = @_[ARG0, ARG1];
                my $retval = undef;
                my $retvalstr = '';
                if( $reason eq 'create' ) {
                    $retval = $_[ARG2];
                    $retvalstr = " (retval=" . $retval . ")"
                }
                my $session = $child->ID;
                my $msg = "_child event '". $reason . "'" . $retvalstr . " from child (id=" . $session . ")";
                dc_log_dbg($msg, $alias_terminal);
            },

            setup_cryptosystems => sub {
                my ($kernel, $heap, $session, $who, $force) = @_[KERNEL, HEAP, SESSION, ARG0, ARG1];
                my ($host, $port, $keyid);

                # find keyid by parsing who as sap to a channel server name
                if ($who) {
                    if ($who =~ /([^\s:]+):([0-9]+)/) {
                        ($host, $port) = ($1, $2);
                        my @names_all = keys %{ $CONF->{node}->{channelserver} };
                        my @names = grep(!/^_/, @names_all);
                        for my $name (@names) {
                            my $h = $CONF->{node}->{channelserver}->{$name}->{host};
                            my $p = $CONF->{node}->{channelserver}->{$name}->{port};

                            $keyid = $CONF->{node}->{channelserver}->{$name}->{key_id}
                                if (($h eq $host) && ($p == $port));
                        }
                    }
                    # if parse failed parse who as channel server name
                    else {
                        $keyid = $CONF->{node}->{channelserver}->{$who}->{key_id};
                    }
                }

                # use _default keyid if no keyid found above and host/port matches
                unless($keyid) {
                    my $h = $CONF->{node}->{channelserver}->{_default}->{host};
                    my $p = $CONF->{node}->{channelserver}->{_default}->{port};

                    $keyid = $CONF->{node}->{channelserver}->{_default}->{key_id}
                        if ((($h eq $host) && ($p == $port)) || !$who);
                }

                # ignore found keyid if force=1
                $keyid = 0 if ($force);

                # check if cryptosystems need to generate key material
                my $no_delay = crypt_base_check_cryptosystems($keyid);

                # if there will be a delay, notify user
                my $ok = 1;
                unless ($no_delay)
                {
                    my $msg_title = $CONF->{product_name} . ' needs to generate new key material';
                    my $msg_delay = "The " . $CONF->{product_name} . " client will generate new public and secret key material.\n\nPlease generate as much system entropy as possible to speedup this process!\nThis process can take up to 5 minutes otherwise...";
                    $ok = $CURSES->dialog(
                        -title   => $msg_title,
                        -message => $msg_delay,
                        -buttons => ['ok', 'cancel'],
                        -buttonalignment => 'right',
                    );

                    return 0 unless ($ok);
                }

                # setup cryptosystems & return resulting keyid which we will use as our identity
                # pass numeric 0 as $key_passphrase to make sure the GPG agent will be used
                $keyid = crypt_base_setup_cryptosystems($host, $keyid, 0)
                    or confess("failed to setup DarkChannel::Crypt::Base crypto systems!");

                return $keyid;
            },

            setup_nickname => sub {
                my ($kernel, $heap, $session, $nickname, $channelserver) = @_[KERNEL, HEAP, SESSION, ARG0, ARG1];
                my $nick_fq = $nickname . '@' . $channelserver;
                my $keyid = 0;

                # check if we need to generate key material for the nickname
                my $no_delay = crypt_base_check_nickname($nickname, $channelserver);

                # if there will be a delay, notify user
                my $ok = 1;
                unless ($no_delay)
                {
                    my $msg_title = $CONF->{product_name} . ' needs to generate new key material';
                    my $msg_delay = "The " . $CONF->{product_name} . " client will generate new public and secret key material.\n\nPlease generate as much system entropy as possible to speedup this process!\nThis process can take up to 5 minutes otherwise...";
                    $ok = $CURSES->dialog(
                        -title   => $msg_title,
                        -message => $msg_delay,
                        -buttons => ['ok', 'cancel'],
                        -buttonalignment => 'right',
                    );

                    return 0 unless ($ok);
                }

                # setup cryptosystems & return resulting keyid which we will use as our identity
                # pass numeric 0 as $key_passphrase to make sure the GPG agent will be used
                $keyid = crypt_base_setup_nickname($nickname, $channelserver, 0)
                    or confess("failed to setup DarkChannel::Crypt::Base nickname!");

                return $keyid;
            },

            do_exit => sub {
                my ($kernel, $heap, $session, $input) = @_[KERNEL, HEAP, SESSION, ARG0];
                my $answer = 1;

                if ($CONF->{settings}->{confirm_quit})
                {
                    # really quit dialog
                    $answer = $heap->dialog(
                        -message => "Really quit?",
                        -buttons => [ 'cancel', 'ok' ],
                        -buttonalignment => 'right',
                        );
                }

                # send 'shutdown' to signal handler if confirmed
                $kernel->post($alias_signalhandler, 'shutdown') if ($answer);
            },

            input_STDERR => sub {
                my ($kernel, $heap, $session, $input) = @_[KERNEL, HEAP, SESSION, ARG0];

                dc_log_dbg($input, 'STDERR');
            },

            darkchannel_CONNECTED => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
                my $sid = $_[ARG0];

                # open channel server notebook page
                $poe_kernel->yield('notebook_page_open', $sid);
            },

            darkchannel_DISCONNECTED => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
                my $sid = $_[ARG0];

                # open channel server notebook page synchroniously to make sure session still exists
                $poe_kernel->call($alias_terminal, 'notebook_page_close', $sid);
            },

            darkchannel_LOG_PROTO => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
                my ($sid, $sap, $message, $prefix) = @_[ARG0, ARG1, ARG2, ARG3];

                # print log message
                dc_poe_term_log($sap, $message, $prefix);
            },

            darkchannel_LOG_CHANNELSERVER => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
                my ($sid, $sap, $message, $prefix) = @_[ARG0, ARG1, ARG2, ARG3];

                # print log message
                dc_poe_term_log($sap, $message, $prefix);
            },

            darkchannel_LOG_CORE => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
                my ($sid, $sap, $message, $prefix) = @_[ARG0, ARG1, ARG2, ARG3];

                # print log message
                dc_poe_term_log('Core', $message, $prefix);
            },

            darkchannel_JOIN => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
                my ($sid, $channel, $join_keyid, $new_channel) = @_[ARG0, ARG1, ARG2, ARG3];
                my $sap = dc_session_data_get($sid, 'sap');
                my $page = $sap . ' ' . $channel;

                # open notebook page if new channel
                $kernel->yield('notebook_page_open', $sid, $channel) if ($new_channel);

                foreach my $id (@{ $join_keyid }) {
                    # print join message
                    my $msg = $id . ' has joined ' . $channel;
                    $kernel->yield('notebook_page_print', $page, $msg);
                }
            },

            darkchannel_PART => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
                my ($sid, $channel, $keyid, $channel_parted) = @_[ARG0, ARG1, ARG2, ARG3];
                my $sap = dc_session_data_get($sid, 'sap');
                my $page = $sap . ' ' . $channel;
                my $msg = $keyid . ' has left ' . $channel;

                # print part message
                $kernel->yield('notebook_page_print', $page, $msg);
            },

            darkchannel_MESSAGE => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
                my ($sid, $recipient, $client_keyid, $nickname_keyid, $message) = @_[ARG0, ARG1, ARG2, ARG3, ARG4];
                my $sap = dc_session_data_get($sid, 'sap');

                # if nick keyid is avaulable, fetch it's uid and show the nickname instad of the client keyid
                my $source = $client_keyid;
                if ($nickname_keyid && crypt_base_key_exists($nickname_keyid)) {
                    my $uids = crypt_base_data_get('pub', $nickname_keyid, 'uid');
                    my $uid = (@{ $uids })[0];
                    $source = $uid->{uid_email};
                    $source =~ s/@.*$//g; # only show nick, no host for the moment
                }

                if ($recipient =~ /^#[a-zA-Z_-]+$/) {
                    my $page = $sap . ' ' . $recipient;
                    my $prefix = '<' . $source . '>';

                    # show MESSAGE
                    $kernel->yield('notebook_page_print', $page, $message, $prefix);
                }
                else {
                    my $notebook = $CURSES->{ui}->{notebook};
                    my $page = $notebook->active_page();
                    my $prefix = '*' . $source . '*';

                    # show MESSAGE
                    $kernel->yield('notebook_page_print', $page, $prefix . ' ' . $message);
                }
            },

            darkchannel_NICK => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
                my ($sid, $channel, $client_keyid, $nick_keyid, $nick_fq) = @_[ARG0, ARG1, ARG2, ARG3, ARG4];
                my $sap = dc_session_data_get($sid, 'sap');
                my $page = $sap . ' ' . $channel;
                my $nick = $nick_fq; $nick =~ s/@.*$//g;
                my $message = $client_keyid . ' is now known as ' . $nick;

                # show NICK change
                $kernel->yield('notebook_page_print', $page, $message);
            },

            darkchannel_PONG => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
                my ($sid, $timestamp, $lag) = @_[ARG0, ARG1, ARG2];

                # decide if we want to display the received pong
                my $cmd_ping = dc_session_data_get($sid, 'command_ping') // '';
                if ($cmd_ping eq 'sent') {
                    my $sap = dc_session_data_get($sid, 'sap');
                    my $msg = 'Received PONG from channel server ' . $sap . ' with a lag of ' . $lag . ' seconds';
                    my $notebook = $CURSES->{ui}->{notebook};
                    my $page = $notebook->active_page();

                    # remove the command_ping state from session
                    dc_session_data_delete($sid, 'command_ping');

                    # print pong message
                    $kernel->yield('notebook_page_print', $page, $msg);
                }
            },

            darkchannel_LIST => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
                my ($sid, $list) = @_[ARG0, ARG1];
                my $msg = 'Available channel list:';

                # format channel list
                if ($list) {
                    $list = "\n" . $list;
                    $list =~ s/\n/\n  /g;
                    $msg .= $list;
                }
                else {
                    $msg = "No such channel";
                }

                my $notebook = $CURSES->{ui}->{notebook};
                my $page = $notebook->active_page();

                # print list message
                $kernel->yield('notebook_page_print', $page, $msg);
            },

            darkchannel_REGISTER => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
                my ($sid, $role, $name, $pubkey_id, $pubkey) = @_[ARG0, ARG1, ARG2, ARG3, ARG4];
                my $sap = dc_session_data_get($sid, 'sap');
                my $namespace = lc($role);
                my $channelserver = (split(/:/, $sap))[0];
                my $name_fq = $name . '@' . $channelserver;
                my $nameref = $CONF->{node}->{$namespace} ? $CONF->{node}->{$namespace}->{$name_fq} : undef;
                my ($msg, $import);

                if ($pubkey) {
                    if ($nameref) {
                        if ($nameref->{key_id} eq $pubkey_id) {
                            if ($nameref->{state} eq 'requested') {
                                # import signed public key for given role
                                $import = crypt_base_key_import($pubkey, $role);

                                # mark name as registered
                                $nameref->{state} = 'registered';

                                # inform user
                                $msg = "The channel server at " . $sap . " granted the identity '" . $name
                                    . "' the role " . $role . " within his name space";
                            }
                        }
                        else {
                            $msg = "Received signed identity (keyid=" . $pubkey_id . ", name=" . $name
                                . " in state '" . $nameref->{state} . "', ignoring";
                        }
                    }
                    else {
                        $msg = "Received signed identity (keyid=" . $pubkey_id . ", name=" . $name
                            . ") not matching requested identity (keyid=" . $nameref->{key_id}
                            . ", name=" . $nameref->{name} . "), ignoring";
                    }
                }
                else {
                    $msg = "Received signed identity (keyid=" . $pubkey_id . ", name=" . $name
                        . ") but did not request it, ignoring";
                }

                my $notebook = $CURSES->{ui}->{notebook};
                my $page = $notebook->active_page();

                # print list message
                $kernel->yield('notebook_page_print', $page, $msg);
            },


            notebook_page_open => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
                my ($who, $channel) = @_[ARG0, ARG1];

                confess("event 'notebook_page_open' without who!") unless($who);

                # who = <sap>|ChannelServer-<sap>|<sid>
                my ($sid, $sap);
                ($sid, $sap) = ($kernel->alias_resolve('ChannelServer-' . $who)->ID, $who) if ($who =~ /^[^-]+:[0-9]+$/);
                ($sid, $sap) = ($kernel->alias_resolve($who)->ID, $1) if ($who =~ /^.+-([^-\s]+:[0-9]+)$/);
                ($sid, $sap) = ($who, dc_session_data_get($who, 'sap')) if ($who =~ /^\d+$/);
                my $alias = $sid ? dc_session_data_get($sid, 'alias') : '';
                my $page = $channel ? $sap . ' ' . $channel : $sap;

                if ($page) {
                    my $req = $alias ? 'for ' . $alias : '';
                    dc_log_dbg("creating notebook page '" . $page . "' " . $req, $alias_terminal);

                    # create notebook page
                    my $notebook = $CURSES->{ui}->{notebook};
                    my $win_server = $notebook->add_page($page);

                    # create notebook textviewer on notebook page
                    my $text_id = 'text-' . $page;
                    my $text_channel = $win_server->add(
                        $text_id    => 'TextViewer',
                        -text       => '',
                        -homeonblur => 0,
                        -wrapping   => 1,
                        -vscrollbar => 'right',
                    );
                    $CURSES->{ui}->{text}->{$page} = $text_channel;
                    $notebook->activate_page($page);
                    $notebook->draw();

                    # setup prompt on this notebook page
                    dc_poe_term_cmd_set_prompt();
                }
                else {
                    dc_log_err("notebook_page_open: could not find page for (who=" . $who .")", $alias_terminal)
                }
            },

            notebook_page_activate => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
                my ($who, $channel) = @_[ARG0, ARG1];

                confess("event 'notebook_page_activate' without who!") unless ($who);

                # who = <sap>|ChannelServer-<sap>|<sid>
                my ($sid, $sap);
                ($sid, $sap) = ($kernel->alias_resolve('ChannelServer-' . $who)->ID, $who) if ($who =~ /^[^-]+:[0-9]+$/);
                ($sid, $sap) = ($kernel->alias_resolve($who)->ID, $1) if ($who =~ /^.+-([^-\s]+:[0-9]+)$/);
                ($sid, $sap) = ($who, dc_session_data_get($who, 'sap')) if ($who =~ /^\d+$/);
                my $alias = $sid ? dc_session_data_get($sid, 'alias') : undef;
                my $page = $channel ? $sap . ' ' . $channel : $sap;

                if ($page) {
                    my $req = $alias ? 'for ' . $alias : '';
                    dc_log_dbg("activating notebook page '" . $page . "' " . $req, $alias_terminal);

                    my $notebook = $CURSES->{ui}->{notebook};
                    $notebook->activate_page($page);
                    $notebook->draw();
                }
                else {
                    dc_log_err("notebook_page_activate: could not find page for (who=" . $who .")", $alias_terminal)
                }
            },

            notebook_page_close => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
                my ($who, $channel) = @_[ARG0, ARG1];

                confess("event 'notebook_page_close' without who!") unless($who);

                # who = <sap>|ChannelServer-<sap>|<sid>
                my ($sid, $sap);
                ($sid, $sap) = ($kernel->alias_resolve('ChannelServer-' . $who)->ID, $who) if ($who =~ /^[^-]+:[0-9]+$/);
                ($sid, $sap) = ($kernel->alias_resolve($who)->ID, $1) if ($who =~ /^.+-([^-\s]+:[0-9]+)$/);
                ($sid, $sap) = ($who, dc_session_data_get($who, 'sap')) if ($who =~ /^\d+$/);
                my $alias = $sid ? dc_session_data_get($sid, 'alias') : undef;
                my $page = $channel ? $sap . ' ' . $channel : $sap;

                if ($page)
                {
                    my $req = $alias ? 'for ' . $alias : '';
                    dc_log_dbg("closing notebook page '" . $page . "' " . $req, $alias_terminal);

                    # delete page from notebook & redraw
                    my $notebook = $CURSES->{ui}->{notebook};
                    $notebook->delete_page($page);

                    # close all channel pages of channels from this channel server
                    my @subpages = $notebook->find_pages('^' . $page);
                    for my $subpage (@subpages) {
                        dc_log_dbg("closing notebook page '" . $subpage . "' " . $req, $alias_terminal);
                        $notebook->delete_page($subpage);
                    }

                    # re-layout and re-draw notebook
                    $CURSES->layout();
                    $CURSES->draw();
                }
                else {
                    dc_log_err("notebook_page_close: could not find page for (who=" . $who .")", $alias_terminal)
                }
            },

            notebook_page_clear => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
                my $page = $_[ARG0];

                confess("event 'notebook_page_clear' without page!") unless($page);

                if ($page)
                {
                    my $text_page = $CURSES->{ui}->{text}->{lc($page)};

                    unless ($text_page) {
                        my $keys = $CURSES->{ui}->{text} ? join(keys %{$CURSES->{ui}->{text}}) : '';
                        dc_log_crit("notebook_page_clear: could not find page '" . lc($page) . "' received as '"
                                    . $page . "' in CURSES->{ui}->{text} which contains currently '"
                                    . $keys . "'", 'Terminal Log');
                    }
                    else {
                        dc_log_dbg("clearing notebook page '" . $page, $alias_terminal);
                        $text_page->text('');
                        $text_page->cursor_to_end();
                        $text_page->parent()->parent()->draw();
                    }
                }
                else {
                    dc_log_err("notebook_page_clear: could not find page (page=" . $page .")", $alias_terminal)
                }
            },

            notebook_page_print => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
                my ($page, $msg, $prefix) = @_[ARG0, ARG1, ARG2];

                confess("event 'notebook_page_print' without page!") unless ($page);
                confess("event 'notebook_page_print' without msg!") unless ($msg);

                # log message to notebook page
                dc_poe_term_log($page, $msg, $prefix);
            },

            leave_curses => sub {
                # leave curses (called synchroniously by SignalHandler)
                $CURSES->leave_curses();
            },

            reset_curses => sub {
                # reset curses (called synchroniously by SignalHandler)
                $CURSES->reset_curses();

                # set focus to user input field
                $CURSES->{ui}->{text}->{input}->focus();
            },

        },
    );


    return dc_poe_term_ui_create($CURSES, $ARGS->{irc_core_support}) ? $alias_terminal : '';
}

sub dc_poe_term_run()
{
    my $this = $CURSES;

    unless ($this->{-no_output}) {
        $this->focus(undef, 1);
        $this->draw;

        Curses::doupdate;
    }

    no warnings "redefine";

    my $modalfocus = \&Curses::UI::Widget::modalfocus;

    # Let modalfocus() be a reentrant into the POE Kernel.  This is stackable,
    # so it should not impact other behaviors, and POE keeps chugging along
    # uneffected.  This is a modal focus without a callback, this method does
    # not return until the modal widget get's cleared out.
    #
    # This is done here so that ->dailog will still work as it did previously.
    # until this is run.  And just in case, we save the old modalfocus
    # definition and redefine it later.
    sub Curses::UI::Widget::modalfocus () {
        my ($this) = @_;

        # "Fake" focus for this object.
        $this->{-has_modal_focus} = 1;
        $this->focus;
        $this->draw;

        push @modal_objects, $this;
        push @modal_callbacks, undef;

        # This is reentrant into the POE::Kernel
        while ( $this->{-has_modal_focus} ) {
            $poe_kernel->loop_do_timeslice;
        }

        $this->{-focus} = 0;

        pop @modal_callbacks;
        pop @modal_objects;

        return $this;
    }

    POE::Kernel->run;

    # the next line of code in Curses::UI::POE::mainloop fails with the following error message:
    #   Can't use string ("Curses::UI::Widget::modalfocus") as a symbol ref  while "strict refs"
    #   in use at /usr/lib64/perl5/vendor_perl/5.20.2/Curses/UI/POE.pm line 224.
    # commenting the line will fix things.
    #
    # Replace previously defined method into the symbol table.
    #*{"Curses::UI::Widget::modalfocus"} = $modalfocus;
}

#
# initialize Node::Client::Term subsystem
#
# dc_poe_term_initialize($mouse_support, $color_support, $irc_core_support)
#

sub dc_poe_term_initialize($$$)
{
    my ($mouse_support, $color_support, $irc_core_support) = @_;

    # initialize this module
    dc_log_dbg("initializing DarkChannel::Node::Client::Term");
    $ARGS->{mouse_support} = $mouse_support;
    $ARGS->{color_support} = $color_support;
    $ARGS->{irc_core_support} = $irc_core_support;

    # initialize term command subsystem
    dc_poe_term_cmd_initialize()
        or confess("Failed to initialize DarkChannel::Node::Client::Term::Command!");

    # initialize poe term client subsystem
    dc_poe_client_initialize()
        or confess("Failed to initialize DarkChannel::Proto::Client!");

    return 1;
}

1;
