#
# Dark Channel Terminal Command Library
#
# Copyright (C) 2015 by DataCore GmbH
#     Amir Guindehi <amir@datacore.ch>
#

package DarkChannel::Node::Client::Term::Command;

use warnings;
use strict;

use Carp;
use Data::Dumper;

use Tie::IxHash;
use Socket;

use DarkChannel::Crypt::Base;

use DarkChannel::Utils::Log;
use DarkChannel::Utils::Text;
use DarkChannel::Utils::SessionStorage;
use DarkChannel::Utils::ChannelStorage;

use DarkChannel::Proto::Client;

use DarkChannel::Node::Client::Term::Debug;
use DarkChannel::Node::Client::Conf;

# Parameters to use POE are not treated as normal imports.
# Rather, they're abbreviated modules to be included along with POE.
use POE qw(Kernel);

use Exporter;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

our $VERSION = 1.00;
our @ISA = qw( Exporter );
our @EXPORT_OK = qw();
our @EXPORT = qw( $TERM_CMD
                  dc_poe_term_cmd_initialize
                  dc_poe_term_cmd_process_usercmd
                  dc_poe_term_cmd_process_userinput
                  dc_poe_term_cmd_set_prompt );

# terminal client commands
my %TERM_CMD_NICK_HELP;
tie %TERM_CMD_NICK_HELP, 'Tie::IxHash';
%TERM_CMD_NICK_HELP = (
    'register'     => { desc => 'register nickname on channel server' },
    );

my %TERM_CMD_KEY_HELP;
tie %TERM_CMD_KEY_HELP, 'Tie::IxHash';
%TERM_CMD_KEY_HELP = (
    'generate'     => { desc => 'generate new key material and store in key ring' },
    'list'         => { desc => 'show current keys in key ring' },
    'remove'       => { desc => 'remove a key from key ring' },
    );

my %TERM_CMD_CLIENT_HELP;
tie %TERM_CMD_CLIENT_HELP, 'Tie::IxHash';
%TERM_CMD_CLIENT_HELP = (
    'dump session' => { desc => 'dump client session state' },
    'dump channel' => { desc => 'dump client channel state' },
    'dump crypto'  => { desc => 'dump client crypto state' },
    'dump conf'    => { desc => 'dump client conf state' },
    );

my %TERM_CMD_SERVER_HELP;
tie %TERM_CMD_SERVER_HELP, 'Tie::IxHash';
%TERM_CMD_SERVER_HELP = (
    'dump session' => { desc => 'dump channel server session state' },
    'dump channel' => { desc => 'dump channel server channel state' },
    'dump crypto'  => { desc => 'dump channel server crypto state' },
    'dump conf'    => { desc => 'dump channel server conf state' },
    );

our %TERM_CMD;
tie %TERM_CMD, 'Tie::IxHash';
our $TERM_CMD = \%TERM_CMD;
%TERM_CMD = (
    'quit'          => { desc => 'quit the DarkChannel client',
                         func => \&dc_poe_term_cmd_quit },
    'help'          => { desc => 'show command help',
                         func => \&dc_poe_term_cmd_help },
    'connect'       => { desc => 'open a new connection to a channel server',
                         func => \&dc_poe_term_cmd_connect },
    'disconnect'    => { desc => 'disconnect a connected channel server',
                         func => \&dc_poe_term_cmd_disconnect },
    'join'          => { desc => 'join a channel on a channel server',
                         func => \&dc_poe_term_cmd_join },
    'part'          => { desc => 'leave a channel on a channel server',
                         func => \&dc_poe_term_cmd_part },
    'ping'          => { desc => 'ping a channel server',
                         func => \&dc_poe_term_cmd_ping },
    'list'          => { desc => 'list existing channels on a channel server',
                         func => \&dc_poe_term_cmd_list },
    'msg'           => { desc => 'send private message',
                         func => \&dc_poe_term_cmd_msg },
    'nick'          => { desc => 'set nickname for channel server',
                         func => \&dc_poe_term_cmd_nick ,
                         help => \%TERM_CMD_NICK_HELP,
                       },
    'clear'         => { desc => 'clear the current page',
                         func => \&dc_poe_term_cmd_clear },
    'suspend'       => { desc => 'suspend the DarkChannel client',
                         func => \&dc_poe_term_cmd_suspend },
    'poe'           => { desc => 'show POE session informations',
                         func => \&dc_poe_term_cmd_poe },
    'key'           => { desc => 'various key management commands',
                         func => \&dc_poe_term_cmd_key,
                         help => \%TERM_CMD_KEY_HELP,
                       },
    'client'        => { desc => 'various client management commands',
                         func => \&dc_poe_term_cmd_client,
                         help => \%TERM_CMD_CLIENT_HELP,
                       },
    'server'        => { desc => 'various client management commands',
                         func => \&dc_poe_term_cmd_server,
                         help => \%TERM_CMD_SERVER_HELP,
                       },
    );

# aliases
my $alias_interpreter= 'Client-Interpreter';
my $alias_terminal= 'Client-Terminal';

sub fmt($$;$)
{
    my ($str, $len, $right) = @_;
    if ($right) {
        $str = ' ' . $str while (length($str) < $len);
        $str = substr(0, length($str) - 1) while (length($str) > $len);
    }
    else {
        $str .= ' ' while (length($str) < $len);
        $str = substr(0, length($str) - 1) while (length($str) > $len);
    }
    return $str;
}

sub cmd_page()
{
    # fetch current notebook page
    my $CURSES = $poe_kernel->alias_resolve($alias_terminal)->get_heap();
    my $notebook = $CURSES->{ui}->{notebook};

    return $notebook->active_page();
}

sub cmd_prompt()
{
    # fetch current notebook page
    my $CURSES = $poe_kernel->alias_resolve($alias_terminal)->get_heap();

    return ($CURSES->{ui}->{label}->{prompt}, $CURSES->{ui}->{prompt});
}

sub cmd_log($;$)
{
    my $msg = shift;
    my $prefix = shift;
    my $page = cmd_page();

    $poe_kernel->post($alias_terminal, 'notebook_page_print', $page, $msg, $prefix);
}

sub cmd_log_err($;$)
{
    my $msg = shift;
    my $prefix = shift;
    my $page = cmd_page();

    $prefix = 'ERR' . ($prefix ? ': ' .$prefix : '');
    $poe_kernel->post($alias_terminal, 'notebook_page_print', $page, $msg, $prefix);
}

sub dc_poe_term_cmd_quit(@)
{
    $poe_kernel->yield('do_exit');
}

sub dc_poe_term_cmd_help($)
{
    my $args = shift;
    my $err = '';
    my $help = '';

    if ($args && (defined($TERM_CMD->{$args}))) {
        if ($TERM_CMD->{$args}->{help}) {
            # generate specific command help with subcommands
            $help = "The following subcommands are available for command '" . $args . "':\n\n";
            $help .= sprintf("%s%s%s\n", fmt('', 2), fmt($_, 20), $TERM_CMD->{$args}->{help}->{$_}->{desc})
                for (keys %{ $TERM_CMD->{$args}->{help} });
        }
        else {
            # generate specific command help without subcommands
            $help = "The following command is available:\n\n";
            $help .= sprintf("%s%s%s\n", fmt('', 2), fmt($args, 20), $TERM_CMD->{$args}->{desc});
        }
    }
    else {
        # generate general command help
        $help = "The following commands are available:\n\n";
        $help .= sprintf("%s%s%s\n", fmt('', 2), fmt($_, 20), $TERM_CMD->{$_}->{desc}) for (keys %{ $TERM_CMD });
    }

    # show help
    cmd_log($help);

    return ($err ? 1:0, $err)
}

#
# /connect - connect to a channel server
#
sub dc_poe_term_cmd_connect($)
{
    my $args = shift;
    my $err = '';

    # check if argument is of 'host:port' form
    if ((not $args)
        || ($args =~ /^([^\s:]+):([0-9]+) ([^\s]+)$/)
        || ($args =~ /^([^\s:]+):([0-9]+)$/)
        || ($args =~ /^([^\s:]+)$/))
    {
        my $host = $1;
        my $port = $2;
        my $name = $3;

        # use '_default' channel server to connect if existing and no argument given
        if (($args eq '') && ($CONF->{node}->{channelserver}->{_default})) {
            # use '_default' channel server sap
            my $h = $CONF->{node}->{channelserver}->{_default}->{host};
            my $p = $CONF->{node}->{channelserver}->{_default}->{port};
            $host = $h if ($h);
            $port = $p if ($p);
        }
        # use '$host' channel server's host/port to connect if just a name given
        if ($host && (not $port) && ($CONF->{node}->{channelserver}->{$host})) {
            my $h = $CONF->{node}->{channelserver}->{$host}->{host};
            my $p = $CONF->{node}->{channelserver}->{$host}->{port};
            $host = $h if ($h);
            $port = $p if ($p);
        }
        # use default port if none given
        $port = 26667 unless ($port);

        # lookup $host to make sure it's an IP address (and use the first address returned)
        unless ($host =~ /^[0-9]/) {
            my @ips = gethostbyname($host);
            $host = $ips[0] if ($#ips >= 0);
        }
        return (1, "could not resolve '" . $host . "'! failed to connect to " . $host . ":" . $port)
            unless ($host);

        # check if channelserver is already connected and fail if so
        my $sap = $host . ':' . $port;
        my $alias= 'ChannelServer-' . $sap;
        if (my $session = $poe_kernel->alias_resolve($sap)) {
            # channel server already connected
            my $sid = $session->ID;
            $err = "Channel Server '".$sap."' is already connected!";

            # switch active notebook page to that channel server
            $poe_kernel->yield('notebook_page_activate', $sid)
        } else {
            # store host/port for later usage in '_default' channel server
            $CONF->{node}->{channelserver}->{_default}->{host} = $host;
            $CONF->{node}->{channelserver}->{_default}->{port} = $port;

            # store host/port for later usage in '$name' channel server if given
            if ($name) {
                $CONF->{node}->{channelserver}->{$name}->{host} = $host;
                $CONF->{node}->{channelserver}->{$name}->{port} = $port;
                $CONF->{node}->{channelserver}->{$name}->{key_id} = 0;   # make sure we get a new key
            }

            # call event 'setup_cryptosystems' which will generate a new key if needed
            my $key_id = $poe_kernel->call($alias_terminal, 'setup_cryptosystems', $sap);

            if ($key_id) {
                # store channel server's keyid in '_default' and $name if given
                $CONF->{node}->{channelserver}->{_default}->{key_id} = $key_id;
                $CONF->{node}->{channelserver}->{$name}->{key_id} = $key_id if ($name);

                # create new DarkChannel::Node::Client::ChannelServer client if key material is ok
                my ($session, $alias) = dc_poe_client_spawn($host, $port, $key_id, $alias_terminal);
            }
        }
    }
    else {
        $err = 'Usage: /connect <host>:<port> [<name>]';
    }

    return ($err ? 1:0, $err)
}

#
# /disconnect - disconnect a channel server
#
sub dc_poe_term_cmd_disconnect($)
{
    my $args = shift;
    my $err = '';

    # check if argument is of 'host:port' form
    if ((not $args)
        || ($args =~ /^([^\s:]+)$/)
        || ($args =~ /^([^\s:]+):([0-9]+)$/)
)
    {
        my $host = $1;
        my $port = $2;

        # no arguments given, use page name to find sap
        if ($args eq '')
        {
            my $page = cmd_page();

            if ($page =~ /^([^:\s]+):([0-9]+)/) {
                $host = $1;
                $port = $2;
            }
            else {
                return (1, 'This command can only be used in a channel server or channel context without argument!');
            }
        }
        # use '$host' channel server's host/port to connect if just a name given
        if ($host && (not $port)) {
            my $h = $CONF->{node}->{channelserver}->{$host}->{host};
            my $p = $CONF->{node}->{channelserver}->{$host}->{port};

            $host = $h if ($h);
            $port = $p if ($p);
        }

        # use default port if none given
        $port = '26667' if ($port);

        # resulting sap
        my $sap = $host . ':' . $port;

        # send disconnect event
        my $alias= 'ChannelServer-' . $sap;
        if ($poe_kernel->alias_resolve($alias)) {
            $poe_kernel->post($alias, 'disconnect')
        }
        else {
            $err = "No channel server connected with address '" . $sap . "'";
        }
    }
    else {
        $err = 'Usage: /disconnect <host>:<port>';
    }

    return ($err ? 1:0, $err)
}

#
# /join - join a channel
#
sub dc_poe_term_cmd_join($)
{
    my $args = shift;
    my $err = '';
    my $page = cmd_page();

    if ($page =~ /([^:\s]+:[0-9]+)/)
    {
        my $sap = $1;
        my $recipient= 'ChannelServer-' . $sap;
        my $recipient_sid = $poe_kernel->alias_resolve($recipient)->ID();
        my $channel = undef;

        # check if argument is of 'channel' form
        if ($args =~ /^(#[a-zA-Z0-9-]+)$/) {
            $channel = $1;
        }
        elsif ($args eq '')
        {
            $channel = '#public';
        }
        else {
            $err = 'Usage: /join <channel>';
        }

        # make interpreter execute a JOIN request to $recipient
        $poe_kernel->post($alias_interpreter, 'cmd_JOIN', $recipient_sid, [ $channel ]) if ($channel);
    }
    else
    {
        $err = 'This command can only be used in a channel server context!';
    }

    return ($err ? 1:0, $err)
}

#
# /leave - part a channel
# /part - part a channel
#
sub dc_poe_term_cmd_part($)
{
    my $args = shift;
    my $err = '';
    my $page = cmd_page();

    # parse page for sap and channel
    my ($sap, $channel);
    if ($page =~ /([^:\s]+:[0-9]+) (#[a-zA-Z-]+)/) {
        ($sap, $channel) = ($1, $2)
    }
    elsif ($page =~ /([^:\s]+:[0-9]+)/) {
        $sap = $1;
    }
    else {
        return (1, 'This command can only be used in a channel server or channel context!');
    }

    # check if argument is of 'channel' form
    if ($args =~ /^(#[a-zA-Z0-9-]+)$/) {
        $channel = $1;
    }
    elsif (($args eq '') && (not $channel))
    {
        $channel = '#public';
    }
    elsif (not $channel) {
        return (1, 'Usage: /part [<channel>]');
    }

    # send events
    my $recipient= 'ChannelServer-' . $sap;
    my $recipient_sid = $poe_kernel->alias_resolve($recipient)->ID();

    if ($channel) {
        # make interpreter execute a PART request to $recipient
        $poe_kernel->post($alias_interpreter, 'cmd_PART', $recipient_sid, [ $channel ]);

        # close notebook page
        $poe_kernel->post($alias_terminal, 'notebook_page_close', $recipient_sid, $channel);
    }

    return ($err ? 1:0, $err);
}

#
# /ping - ping a channel server
#
sub dc_poe_term_cmd_ping($)
{
    my $args = shift;
    my $err = '';
    my $page = cmd_page();

    # parse page for sap and channel
    my ($sap, $channel);
    if ($page =~ /([^:\s]+:[0-9]+) (#[a-zA-Z-]+)/) {
        ($sap, $channel) = ($1, $2)
    }
    elsif ($page =~ /([^:\s]+:[0-9]+)/) {
        $sap = $1;
    }
    else {
        return (1, 'This command can only be used in a channel server or channel context!');
    }

    # check if no arguments given
    if ($args ne '') {
        return (1, 'Usage: /ping');
    }

    # send events
    my $recipient= 'ChannelServer-' . $sap;
    my $recipient_sid = $poe_kernel->alias_resolve($recipient)->ID();

    # remember we requested a PING which will result in a PONG
    dc_session_data_set($recipient_sid, 'command_ping', 'sent');

    # make interpreter execute a PING request to $recipient
    $poe_kernel->post($alias_interpreter, 'cmd_PING', $recipient_sid, []);

    return ($err ? 1:0, $err);
}

#
# /msg - send private message
#
sub dc_poe_term_cmd_msg($)
{
    my $args = shift;
    my $err = '';
    my $page = cmd_page();

    # parse page for sap and channel
    my ($sap, $channel);
    if ($page =~ /([^:\s]+:[0-9]+) (#[a-zA-Z-]+)/) {
        ($sap, $channel) = ($1, $2)
    }
    elsif ($page =~ /([^:\s]+:[0-9]+)/) {
        $sap = $1;
    }
    else {
        return (1, 'This command can only be used in a channel server or channel context!');
    }

    # parse command argument
    my ($recipient_keyid, $message);
    if ($args =~ /^([^\s]+) (.+)$/) {
        ($recipient_keyid, $message) = ($1, $2);
    }
    else {
        return (1, 'Usage: /msg <recipient> <message>');
    }

    my $recipient= 'ChannelServer-' . $sap;
    my $recipient_sid = $poe_kernel->alias_resolve($recipient)->ID();

    # make interpreter execute a RELAY MESSAGE request to $recipient
    my $success = $poe_kernel->call($alias_interpreter, 'cmd_RELAY_MESSAGE',
                                        $recipient_sid, [ $recipient_keyid, $message ]);

    if ($success > 0) {
        # local echo to notebook page
        $message = '*' . $recipient_keyid . '* ' . $message;
        cmd_log($message);
    }
    else
    {
        # show error
        cmd_log_err('Failed to send message!');
    }
    return;
}

#
# /list - list channels on a channel server
#
sub dc_poe_term_cmd_list($)
{
    my $args = shift;
    my $err = '';
    my $page = cmd_page();

    # parse page for sap and channel
    my ($sap, $channel);
    if ($page =~ /([^:\s]+:[0-9]+) (#[a-zA-Z-]+)/) {
        ($sap, $channel) = ($1, $2)
    }
    elsif ($page =~ /([^:\s]+:[0-9]+)/) {
        $sap = $1;
    }
    else {
        return (1, 'This command can only be used in a channel server or channel context!');
    }

    # check if argument is of 'channel' pattern form
    my $channel_pattern = '';
    if ($args =~ /^(#?[a-zA-Z0-9-]+)$/) {
        $channel_pattern = $1;
    }
    elsif ($args ne '') {
        return (1, 'Usage: /ping');
    }

    # send events
    my $recipient= 'ChannelServer-' . $sap;
    my $recipient_sid = $poe_kernel->alias_resolve($recipient)->ID();

    # make interpreter execute a LIST request to $recipient
    $poe_kernel->post($alias_interpreter, 'cmd_LIST', $recipient_sid, [ $channel_pattern ]);

    return ($err ? 1:0, $err);
}

#
# /nick - change nick name
#
sub dc_poe_term_cmd_nick($)
{
    my $args = shift;
    my $err = '';
    my $page = cmd_page();

    # parse page for sap and channel
    my ($sap, $channel);
    if ($page =~ /([^:\s]+:[0-9]+) (#[a-zA-Z-]+)/) {
        ($sap, $channel) = ($1, $2)
    }
    elsif ($page =~ /([^:\s]+:[0-9]+)/) {
        $sap = $1;
        return (1, 'This command can only be used in a channel context!')
            unless ($args =~ /^register [^\s]+$/);
    }
    else {
        return (1, 'This command can only be used in a channel server or channel context!');
    }

    # check if no arguments given
    my $nick = '';
    my $register = 0;
    if ($args eq '') {
        return (1, 'Usage: /nick [register] <nick>');
    }
    elsif ($args =~ /^register ([a-zA-Z_-]+)$/) {
        $nick = $1;
        $register = 1;
    }
    elsif ($args =~ /^([a-zA-Z_-]+)$/) {
        $nick = $1;
    }
    else {
        return (1, 'Usage: /nick [register] <nick>');
    }

    # send event data
    my $recipient= 'ChannelServer-' . $sap;
    my $recipient_sid = $poe_kernel->alias_resolve($recipient)->ID();

    # check for nick name
    my $channelserver = (split(/:/, $sap))[0];
    my $nick_fq = $nick . '@' . $channelserver;
    my $nickref = $CONF->{node}->{nickname} ? $CONF->{node}->{nickname}->{$nick_fq} : undef;

    if ($register && $nickref) {
        return (1, "The nickname '" . $nick . "' is already registered with this channel server!"
                . "\nUse: /nick " . $nick . " to use the nickname in a channel!");
    }
    if (!$register && !$nickref) {
        return (1, "The nickname '" . $nick . "' is not yet registered with this channel server!"
                . "\nUse: /nick register " . $nick . " to register this nickname!");
    }

    # call event 'setup_nickname' which will generate a new key if needed
    my $key_id = $poe_kernel->call($alias_terminal, 'setup_nickname', $nick, $channelserver);

    if ($key_id) {
        if ($register) {
            # store channel server's keyid in '_default' and $name if given
            $CONF->{node}->{nickname}->{$nick_fq}->{key_id} = $key_id;
            $CONF->{node}->{nickname}->{$nick_fq}->{name} = $nick;
            $CONF->{node}->{nickname}->{$nick_fq}->{state} = 'requested';
            $CONF->{node}->{nickname}->{$nick_fq}->{channelserver} = $sap;

            # make interpreter execute a REGISTER request (role nickname) to $recipient
            my $role = 'nickname';
            $poe_kernel->post($alias_interpreter, 'cmd_REGISTER', $recipient_sid, [ $nick, $key_id, $role ]);
        }
        else {
            if ($nickref->{state} ne 'registered') {
                return (1, "Registration of the nickname '" . $nick
                        . "' is not yet finished, please wait for it to complete!");
            }

            # make interpreter execute a RELAY NICK request to $recipient
            my $success = $poe_kernel->call($alias_interpreter, 'cmd_RELAY_NICK',
                                            $recipient_sid, [ $channel, $key_id, $nick ]);

            if ($success > 0) {
                # make nickname active
                my $ckey = $recipient_sid . ':' . $channel;
                dc_channel_data_set($ckey, 'nickname', $nickref);

                # inform user on nickname change
                my $msg = 'You are now known as ' . $nick;
                cmd_log($msg);

                # reflect nickname change in prompt on this notebook page
                dc_poe_term_cmd_set_prompt();
            }
        }
    }
    else {
        $err = "Command canceled";
    }

    return ($err ? 1:0, $err);
}

#
# /clear - clear a notebook page
#
sub dc_poe_term_cmd_clear($)
{
    my $args = shift;
    my $err = '';
    my $page = cmd_page();

    # check if argument is of 'channel' form
    if ($args ne '') {
        return (1, 'Usage: /ping');
    }

    # make interpreter execute a PART request to $recipient
    $poe_kernel->yield('notebook_page_clear', $page);

    return ($err ? 1:0, $err);
}

#
# /suspend - suspend terminal client
#
sub dc_poe_term_cmd_suspend($)
{
    my $args = shift;
    my $alias_signalhandler = 'Client-SignalHandler';
    my $err = '';

    # send suspend signal TSTP to terminal
    $poe_kernel->signal($alias_signalhandler, 'TSTP');

    return ($err ? 1:0, $err)
}

#
# /poe - dump client POE session information
#
sub dc_poe_term_cmd_poe($)
{
    my $args = shift;
    my $err = '';

    # return error if called with arguments
    return (1, 'this command does not accept arguments') if ($args);

    # collect & render debug data
    my $stats = dc_term_debug_collect();
    my $dataref = dc_term_debug_render($stats);
    my $data = join("\n", @{$dataref});

    # show debug data
    # XXX: TODO: show data on current notebook page and not on debug only
    cmd_log($data, 'POE Peak');

    return (0, '');
}


#
# /key - various key management commands
#
sub dc_poe_term_cmd_key($)
{
    my $args = shift;
    my $err = '';

    # return error if not called with arguments
    return (1, 'this command needs arguments, see /help key') if (not $args);

    # collect & render debug data
    my $data = '';

    # 'gen' | 'generate'
    if (($args =~ /^gen ([^\s]+)$/)
        || ($args =~ /^generate ([^\s]+)$/)) {
        my $name = $1;
        my $keyid = $poe_kernel->call($alias_terminal, 'setup_cryptosystems', $name);

        if ($keyid) {
            cmd_log("generated new key " . $keyid . " for channel server '" . $name . "'");
            return (0, '');
        }
        return (1, "failed to generate a new key for channel server '" . $name . "'");
    }
    elsif (($args =~ /^ls$/)
           || ($args =~ /^list$/)
           || ($args =~ /^ls ([^\s]+)$/)
           || ($args =~ /^list ([^\s]+)$/))
    {
        my $key = $1 // '';
        my $pub = crypt_base_key_information_hash($key, 0);
        my $sec = crypt_base_key_information_hash($key, 1);

        for my $id (keys %{$sec}) {
            $pub->{$id}->{type} = 'pub/sec' if ($pub->{$id});
            $pub->{$id} = $sec->{$id} unless ($pub->{$id});
        }

        my $default;
        for my $tag (keys %{$CONF->{node}->{channelserver}}) {
            my $id = $CONF->{node}->{channelserver}->{$tag}->{key_id};
            next unless($id && $pub->{$id});
            $pub->{$id}->{tag} = $tag;
            $default = $pub->{$id} if ($tag eq '_default');
        }
        $default->{default} = '(default)' if ($default);

        for my $comment ('Nickname', 'Client', 'Channel Server') {
            for my $id (sort keys %{$pub}) {
                my $k = $pub->{$id};
                my $c = $k->{comment};
                my $tag = $k->{tag} // '';
                my $default = $k->{default} // '';
                next if ($comment ne $c);
                $data .= fmt('', 2) . fmt($id, 20) . fmt($k->{type}, 9) . fmt($k->{trust}, 8)
                    # . fmt($k->{name}, 28)
                    . fmt('', 2) . fmt($k->{comment}, 16) . fmt($k->{email}, 30, 1) . '  '
                    . $tag . '  ' . $default . "\n";
            }
        }

        # show data
        cmd_log("Key Material:\n\n" . $data);
        return (0, '');
    }

    return (1, "unknown sub command '" . $args . "'");
}

#
# /client - various client commands
#
sub dc_poe_term_cmd_client($)
{
    my $args = shift;
    my $err = '';

    # return error if not called with arguments
    return (1, 'this command needs arguments, see /help client') if (not $args);

    # collect & render debug data
    my $data = '';
    $data = Dumper(dc_session_data_get()) if ($args eq 'dump session');
    $data = Dumper(dc_channel_data_get()) if ($args eq 'dump channel');
    $data = Dumper(crypt_base_data_get()) if ($args eq 'dump crypto');
    $data = Dumper($CONF)                 if ($args eq 'dump conf');

    if ($data) {
        # replace VAR1 with sensible variable name for dump commands
        if ($args =~ /^dump ([^\s]+)$/) {
            my $key = uc($1); $data =~ s/VAR1/$key/g;
        }
        # shorten gpg data when dumping crypto data
        $data = dc_text_transform_shorten_gpg_blocks($data);

        # show dump data
        cmd_log($data, 'State Dump');
        return (0, '');
    }

    return (1, "unknown sub command '" . $args . "'");
}

#
# /channelserver - various channel server commands
#
sub dc_poe_term_cmd_server($)
{
    my $args = shift;
    my $page = cmd_page();
    my $err = '';

    # return error if not called with arguments
    return (1, 'this command needs arguments, see /help channelserver') if (not $args);

    # collect & render debug data
    my $cmd = '';
    $cmd = 'DUMP SESSION' if ($args eq 'dump session');
    $cmd = 'DUMP CHANNEL' if ($args eq 'dump channel');
    $cmd = 'DUMP CRYPTO' if ($args eq 'dump crypto');
    $cmd = 'DUMP CONF' if ($args eq 'dump conf');

    # parse page for sap and channel
    my ($sap, $channel);
    if ($page =~ /([^:\s]+:[0-9]+) (#[a-zA-Z-]+)/) {
        ($sap, $channel) = ($1, $2)
    }
    elsif ($page =~ /([^:\s]+:[0-9]+)/) {
        $sap = $1;
    }
    else {
        return (1, 'This command can only be used in a channel server or channel context!');
    }

    if ($cmd)
    {
        # send command
        my $recipient= 'ChannelServer-' . $sap;
        my $recipient_sid = $poe_kernel->alias_resolve($recipient)->ID();

        # make interpreter execute a CHANNELSERVER request to $recipient
        $poe_kernel->post($alias_interpreter, 'cmd_CHANNELSERVER', $recipient_sid, [ $cmd ]) if ($cmd);

        return (0, '');
    }

    return (1, "unknown sub command '" . $args . "'");
}

sub dc_poe_term_cmd_process_usercmd($$)
{
    my ($cmd, $args) = @_;

    if ($CONF->{log}->{log_dbg_session_terminal}) {
        my $msg = "received user command: cmd='" . $cmd . "'";
        $msg .= ", args='" . $args ."'" if ($args);
        dc_log_dbg($msg, $alias_terminal);
    }

    # execute command if command function exists
    if (defined($TERM_CMD->{$cmd}) && (defined($TERM_CMD->{$cmd}->{func})))
    {
        my $cmd_func = $TERM_CMD->{$cmd}->{func};
        my ($ret, $err) = $cmd_func->($args);

        cmd_log_err($err, $alias_terminal) if ($ret);
        return;
    }

    cmd_log_err("Unknown command /" . $cmd ."\nUse /help to see available commands...");
    return;
}

sub dc_poe_term_cmd_process_userinput($)
{
    my ($input) = @_;
    my $page = cmd_page();

    # XXX: TODO: check if connected to a channel server and on a channel page

    # don't accept empty lines
    return unless($input);

    # check current notebook page for channel page
    if ($page =~ /([^:\s]+:[0-9]+) (#[^\s]+)/)
    {
        my ($sap, $channel) = ($1, $2);

        my $recipient= 'ChannelServer-' . $sap;
        my $recipient_sid = $poe_kernel->alias_resolve($recipient)->ID();

        # make interpreter execute a RELAY request to $recipient
        my $success = $poe_kernel->call($alias_interpreter, 'cmd_RELAY_MESSAGE',
                                        $recipient_sid, [ $channel, $input ]);

        if ($success > 0) {
            # local echo to channel notebook page
            cmd_log($input);
        }
    }
    else
    {
        # show error
        cmd_log_err('You can only talk in a channel context!');
    }
    return;
}

sub dc_poe_term_cmd_set_prompt()
{
    my $page = cmd_page();
    my ($prompt, $prompt_template) = cmd_prompt();

    my $nick = 'none';
    my ($sap, $channel);

    # fetch sap & channel
    if ($page =~ /([^:\s]+:[0-9]+) (#[^\s]+)/) {
        ($sap, $channel) = ($1, $2);
    }
    elsif ($page =~ /([^:\s]+:[0-9]+)/) {
        $sap = $1;
    }

    # figure out nickname
    if ($sap) {
        # fetch channelserver sid
        my $recipient= 'ChannelServer-' . $sap;
        my $recipient_sid = $poe_kernel->alias_resolve($recipient)->ID();

        # by default, use key_id as nickname
        my $key_id = dc_session_data_get($recipient_sid, 'key_id');
        $nick = $key_id if ($key_id);

        # if we found a channel, check for nickname
        if ($channel) {
            my $ckey = $recipient_sid . ':' . $channel;
            my $nickref = dc_channel_data_get($ckey, 'nickname');
            $nick = $nickref->{name} if ($nickref);
        }
    }

    # set new prompt
    $prompt->text(sprintf($prompt_template, $nick));
    $prompt->parent()->parent()->draw();
}

#
# initialize Node::Client::Term subsystem
#
# dc_poe_term_initialize()
#

sub dc_poe_term_cmd_initialize()
{
    # initialize this module
    dc_log_dbg("initializing DarkChannel::Node::Client::Term::Command");

    return 1;
}

1;
