#
# Dark Channel Curses::UI:POE Terminal POE Library
#
# Copyright (C) 2015 by DataCore GmbH
#     Amir Guindehi <amir@datacore.ch>
#

package DarkChannel::Node::Client::Term::Debug;

use warnings;
use strict;

use Carp;
use Data::Dumper;
use POSIX qw(strftime);

use DarkChannel::Utils::Log;

# Parameters to use POE are not treated as normal imports.
# Rather, they're abbreviated modules to be included along with POE.
use POE qw(API::Peek Session);

use Exporter;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

our $VERSION = 1.00;
our @ISA = qw( Exporter );
our @EXPORT_OK = qw();
our @EXPORT = qw( dc_term_debug_collect
                  dc_term_debug_render );

#
# collect information on POE Process, POE Kernel and POE Sessions and return them as a hashref
#
# $stats = dc_term_debug_collect()
#

sub dc_term_debug_collect
{
    my $poe_api = POE::API::Peek->new;
    my $now = time;

    # collect general data about the current process
    my @times = times;
    my @pwent = getpwuid(int $>);
    my $egid  = (split / /, $))[0];
    my @grent = getgrgid(int $egid);

    my %general = (
        process => {
            pid     => $$,
            uid     => $>,
            gid     => $egid,
            user    => $pwent[0],
            group   => $grent[0],
        },
        resource => {
            utime_self  => $times[0],
            utime_chld  => $times[2],
            stime_self  => $times[1],
            stime_chld  => $times[3],
        },
        poe => {
            sessions    => $poe_api->session_count,
            handles     => $poe_api->handle_count,
            loop        => $poe_api->which_loop,
        },
        );

    # collect information about the sessions
    my $kernel_id = $poe_kernel->ID;
    my @sessions;

    for my $session ($poe_api->session_list) {
        push @sessions, {
            $session->ID eq $kernel_id ? (
                id        => 0,
                aliases   => "[POE::Kernel] id=".$session->ID,
                ) : (
                id        => $session->ID,
                aliases   => join(",", $poe_api->session_alias_list($session)),
                ),
                memory_size   => $poe_api->session_memory_size($session),
                refcount      => $poe_api->get_session_refcount($session),
                events_to     => $poe_api->event_count_to($session),
                events_from   => $poe_api->event_count_from($session),
        };
    }

    @sessions = sort { $a->{id} <=> $b->{id} } @sessions;

    # collect information about the events
    my @events;

    for my $event ($poe_api->event_queue_dump) {
        push @events, {
            id          => $event->{ID},
            name        => $event->{event},
            type        => $event->{type},
            priority    => $event->{priority} > $now ?
                $event->{priority} - $now : $event->{priority},
                source      => $event->{source}->ID,
                destination => $event->{destination}->ID,
        }
    }

    # create the final hash
    my %stats = (
        general     => \%general,
        sessions    => \@sessions,
        events      => \@events,
        );

    return \%stats
}

#
# render collected data as text
#
# $text = dc_term_debug_render()
#

sub dc_term_debug_render
{
    my $stats = shift;
    my $proc    = $stats->{general}{process};
    my $rsrc    = $stats->{general}{resource};

    my $session_head    = "%5s  %6s  %8s  %6s  %8s  %-40s";
    my $session_row     = "%5d  %6s  %8d  %6d  %8d  %-40s";
    my @session_cols    = qw(ID Memory Refcount EvtsTo EvtsFrom Aliases);

    my $event_head      = "%5s  %-17s %4s %5s %5s  %-40s";
    my $event_row       = "%5d  %-17s %4d %5d %5d  %-40s";
    my @event_cols      = qw(ID Type Pri Src Dest Name);

    confess("required argument 'stats' missing!") if (not $stats);

    my @data = (
        "Perl Object Environment API Peak",
        "",
        "Running process:",
        "",
        "   PID: $proc->{pid}",
        "   UID: $proc->{uid} ($proc->{user})",
        "   GID: $proc->{gid} ($proc->{group})",
        "",
        "Resource usage:",
        "",
        "   User:   $rsrc->{utime_self} sec (+$rsrc->{utime_chld} sec)",
        "   System: $rsrc->{stime_self} sec (+$rsrc->{stime_chld} sec)",
        "",
        "POE configuration:",
        "",
        "   Loop type: $stats->{general}{poe}{loop}",
        "",
        "POE status:",
        "",
        "   Active sessions: $stats->{general}{poe}{sessions} total",
        "   Active handles:  $stats->{general}{poe}{handles} total",
    );

    push(@data, '', "POE session list:", '');
    push(@data, sprintf($session_head, @session_cols));
    push(@data, sprintf($session_row,
                        $_->{id},
                        dc_poe_term_poe_human_size($_->{memory_size}),
                        $_->{refcount},
                        $_->{events_to},
                        $_->{events_from},
                        $_->{aliases}))
         for (@{$stats->{sessions}});

    if (@{$stats->{events}})
    {
        push(@data, '', "POE event queue:", '');
        push(@data, sprintf($event_head, @event_cols));
        push(@data, sprintf($event_row,
                            $_->{id},
                            $_->{type},
                            $_->{priority},
                            $_->{source},
                            $_->{destination},
                            $_->{name}))
            for (@{$stats->{events}});
    }

    return \@data;
}

sub dc_poe_term_poe_human_size {
    my ($size) = @_;

    return $size if $size < 100_000;

    my $unit;
    for (qw< K M G >) {
        $size = int($size / 1024);
        $unit = $_;
        last if $size < 1024;
    }

    return $size.$unit;
}
