#
# Dark Channel Curses::UI:POE Terminal Library
#
# Copyright (C) 2015 by DataCore GmbH
#     Amir Guindehi <amir@datacore.ch>
#

package DarkChannel::Node::Client::Term::UI;

use warnings;
use strict;

use Carp;
use Data::Dumper;

use DarkChannel::Utils::Log;
use DarkChannel::Utils::SessionStorage;

use DarkChannel::Node::Client::Term::Notebook;
use DarkChannel::Node::Client::Term::History;

use DarkChannel::Node::Client::Conf;

# POE Debugging
#sub POE::Kernel::ASSERT_DEFAULT () { 1 }
#sub POE::Kernel::ASSERT_EVENTS  () { 1 }

# Note: POE's default event loop uses select().
# See CPAN for more efficient POE::Loop classes.
#
# Parameters to use POE are not treated as normal imports.
# Rather, they're abbreviated modules to be included along with POE.
use POE qw(Kernel);

use Curses;
#use Curses::UI::POE;

use Exporter;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

our $VERSION = 1.00;
our @ISA = qw( Exporter );
our @EXPORT_OK = qw();
our @EXPORT = qw( dc_poe_term_ui_initialize
                  dc_poe_term_ui_create );

sub dc_poe_term_ui_create($$)
{
    my $CURSES = shift;
    my $irc_core_support = shift;

    my $product = $CONF->{product_name};
    my $service = $CONF->{service_name};

    # window: output
    my $win_output = $CURSES->add(
        'output', 'Window',
        -border => 1,
        -padtop => 1, # leave space for the menu
        -ipad   => 0,
        -title  => join(' :', $product, $service),
    );
    $CURSES->{ui}->{window}->{output} = $win_output;

    # Prompt Template
    my $prompt_template = '[%18s] $';
    $CURSES->{ui}->{prompt} = $prompt_template;

    # Prompt Label
    my $prompt = sprintf($prompt_template, 'none');
    my $label_prompt = $win_output->add(
        'prompt' => 'Label',
        -y       => -1,
        -text    => $prompt,
    );
    $CURSES->{ui}->{label}->{prompt} = $label_prompt;

    # Notebook
    my $notebook = $win_output->add(
        'notebook', 'DarkChannel::Node::Client::Term::Notebook', # HOTFIX for 'Notebook',
        -padbottom => 1, # leave space for input
        -ipad      => 0,
    );
    $CURSES->{ui}->{notebook} = $notebook;

    # textviewer: Input
    my $text_input = $win_output->add(
        'input'     => 'TextEntry',
        -x          => length($prompt_template) + 18 - 3,
        -y          => -1,
        -readonly   => 0,
        -showlines  => 0,
        -maxlines   => 1,
        -homeonblur => 0,
        -text       => '',
    );
    $CURSES->{ui}->{text}->{input} = $text_input;

    # output notebook page: Log
    my $win_log = $notebook->add_page('Log');

    # textviewer: Log
    my $text_log = $win_log->add(
        'text-Log'  => 'TextViewer',
        -text       => "Logger startup...\n",
        -homeonblur => 0,
        -wrapping   => 1,
        -vscrollbar => 'right',
    );
    $CURSES->{ui}->{text}->{log} = $text_log;

    # output notebook page: Core
    if ($irc_core_support) {
        my $win_core = $notebook->add_page('Core');

        # textviewer: Core
        my $text_core = $win_core->add(
            'text-Core'  => 'TextViewer',
            -text       => "Core startup...\n",
            -homeonblur => 0,
            -wrapping   => 1,
            -vscrollbar => 'right',
        );

        $CURSES->{ui}->{text}->{core} = $text_core;
    }

    # output notebook page: Server
    #my $win_server1 = $notebook->add_page('Server 1');
    #$win_server1->add(
    #    undef, 'Label',
    #    -x    => 15,
    #    -y    => 6,
    #);
    #
    # textviewer: Server1
    #my $text_server1 = $win_server1->add(
    #    'text-Server1' => 'TextViewer',
    #    -text          => 'Server 1 startup...',
    #    -homeonblur => 0,
    #    -vscrollbar => 'right',
    #);
    #$CURSES->{ui}->{text}->{server1} = $text_server1;

    # Menu
    my $menu = $CURSES->add (
        'menu','Menubar',
        -fg   => "white",
        -bg   => "blue",
        -menu => [
            { -label => 'File',
              -submenu => [
                   { -label => 'Connect...    ^C', -value => \&dc_poe_term_cmd_quit },
                   { -label => 'Quit          ^Q', -value => \&dc_poe_term_cmd_quit },
                  ],
            },
            { -label => 'Help',
              -submenu => [
                   { -label => 'About editor    ', -value => \&dc_poe_term_cmd_quit }
                  ],
            },
        ],
    );
    $CURSES->{ui}->{menu} = $menu;

    # focus on user input box
    $text_input->focus();

    # key binding for notebook: ctrl-]/[
    #$notebook->set_binding(sub { $notebook->active_pageup(); }, "\c[");                                        # ctrl-[
    #$notebook->set_binding(sub { $notebook->active_pagedown(); }, "\c]");                                      # ctrl-]

    # key bindings for input: UP
    $text_input->set_binding(sub { $notebook->active_lineup(); }, 561);                                        # ctrl-UP
    $text_input->set_binding(sub { $notebook->active_pageup(); }, 559);                                        # meta-UP
    $text_input->set_binding(sub { $notebook->active_pageup(); }, "\c[");                                      # ctrl-[

    # key bindings for input:DOWN
    $text_input->set_binding(sub { $notebook->active_linedown(); }, 520);                                      # ctrl-DOWN
    $text_input->set_binding(sub { $notebook->active_pagedown(); }, 518);                                      # meta-DOWN
    $text_input->set_binding(sub { $notebook->active_pagedown(); }, "\c]");                                    # ctrl-]

    # XXX: TODO: add key bindings for ctrl-left/right, meta-left/right -> cursor left/right in notebook

    # key bindings for input: TAB/shift-TAB -> focus next/prev notebook page
    $text_input->set_binding(sub { $notebook->activate_page($notebook->next_page()); &DarkChannel::Node::Client::Term::Command::dc_poe_term_cmd_set_prompt; }, "\t");                 # TAB
    $text_input->set_binding(sub { $notebook->activate_page($notebook->prev_page()); &DarkChannel::Node::Client::Term::Command::dc_poe_term_cmd_set_prompt; }, KEY_BTAB());           # shift-TAB

    # key bindings for input: cursor up/down for history
    $text_input->set_binding(sub { dc_poe_term_cmd_history_prev($_[0]); }, KEY_UP());                          # cursor-UP
    $text_input->set_binding(sub { dc_poe_term_cmd_history_next($_[0]); }, KEY_DOWN());                        # cursor-DOWN

    # key binding for text_input
    # never loose focus again, but needs to bind everything to $text_input
    #$text_input->clear_binding('loose-focus');

    # key binding for ENTER
    $text_input->set_binding(\&DarkChannel::Node::Client::Term::dc_poe_term_process_user_input, KEY_ENTER());  # ENTER

    # global key bindings
    $CURSES->set_binding(sub { $_[0]->layout; $_[0]->draw; }, "\cL");                                          # ctrl-l

    # global key bindings
    $CURSES->set_binding(sub { shift()->root->focus('menu') }, "\cX");                                         # ctrl-x
    $CURSES->set_binding(\&DarkChannel::Node::Client::Term::Command::dc_poe_term_cmd_quit, "\cQ");             # ctrl-q
    $CURSES->set_binding(\&DarkChannel::Node::Client::Term::Command::dc_poe_term_cmd_quit, "\cC");             # ctrl-c
    $CURSES->set_binding(\&DarkChannel::Node::Client::Term::Command::dc_poe_term_cmd_suspend, "\cZ" );         # ctrl-z

    # XXX: FIX: howto bind ctrl-\ ?!
    #$CURSES->set_binding(\&dc_poe_term_do_quit, '\c\\\\');                                                    # ctrl-\

    return $CURSES;
}

1;
