#
# Dark Channel POE::Component::Client::TCP Client Library
#
# Copyright (C) 2015 by DataCore GmbH
#     Amir Guindehi <amir@datacore.ch>
#

package DarkChannel::Proto::Client;

use warnings;
use strict;

use Carp;
use Data::Dumper;
use POSIX qw(strftime);

use DarkChannel::Utils::Log;
use DarkChannel::Utils::SessionStorage;

use DarkChannel::Proto::Client::Request;
use DarkChannel::Proto::Client::Response;
use DarkChannel::Proto::V1;

use DarkChannel::Node::Client::Conf;

# POE Debugging
#sub POE::Kernel::ASSERT_DEFAULT () { 1 }
#sub POE::Kernel::ASSERT_EVENTS  () { 1 }
#sub POE::Kernel::CATCH_EXCEPTIONS () { 0 }

# Note: POE's default event loop uses select().
# See CPAN for more efficient POE::Loop classes.
#
# Parameters to use POE are not treated as normal imports.
# Rather, they're abbreviated modules to be included along with POE.
use POE qw(Component::Client::TCP);
use Curses::UI::POE;

use Exporter;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

our $VERSION = 1.00;
our @ISA = qw( Exporter );
our @EXPORT_OK = qw();
our @EXPORT = qw( dc_poe_client_initialize
                  dc_poe_client_spawn
                  dc_poe_client_aliases );

# set by initialize
my $alias_interpreter;

sub c_log($;$)
{
    my ($msg, $prefix) = @_;
    my $session = $poe_kernel->get_active_session();
    my $sid = $session->ID;
    my $heap = $session->get_heap();
    my $alias = $heap->{alias};
    my $alias_customer = $heap->{alias_customer};
    my $page = ($alias =~ /.+-([^:\s]+:[0-9]+)$/) ? $1 : undef;
    my @data = ($sid, $page, $msg, $prefix);

    return 0 unless($page);

    if (not $poe_kernel->post($alias_customer, 'darkchannel_LOG_CHANNELSERVER', @data))
    {
        dc_log_err("failed to send post('" . $alias_customer . "' , 'darkchannel_LOG_CHANNELSERVER', '"
                   . join(', ',@data) . "')");
        dc_log_err("error message: $!");
        return 0;
    }
    return 1;
}
sub dc_poe_client_process_response($$)
{
    my $sid = shift;
    my $heap = shift // confess("No heap in 'dc_poe_client_process_response'");
    my $alias = $heap->{alias};
    my $response = $heap->{input_buf}; chomp($response);

    # call interpreter process function
    if (not $poe_kernel->post($alias_interpreter, 'process_response', $sid, $response))
    {
        dc_log_err("failed to send post('" . $alias_interpreter . "' , 'process_response', " . $sid . ", ...)", $alias);
        dc_log_err("error message: $!", $alias);
    }

    # clear cmd
    $heap->{input_buf} = '';
}

sub dc_poe_client_process_input($$$)
{
    my $sid = shift;
    my $heap = shift;
    my $input = shift;
    my $alias = $heap->{alias};

    if (not ($input =~ /^\.$/))
    {
        # cmd not yet finished
        dc_log_dbg("'$input'", $alias . ': Input') if ($CONF->{log}->{log_dbg_input});
        $heap->{input_buf} .= $input . "\n";
    }
    else
    {
        # cmd completely received, process cmd
        dc_poe_client_process_response($sid, $heap);
    }
}

sub dc_poe_client_spawn($$$$;$$)
{
    my ($connect_addr, $connect_port, $key_id, $alias_customer, $alias_prefix, $opaque_sdata) = @_;

    my $sap = $connect_addr . ':' . $connect_port;
    my $alias_channelserver = ($alias_prefix // '') . 'ChannelServer-' . $sap;
    my $debug = $CONF->{log}->{log_dbg_session_channelserver_tcp};
    my $ping_interval = 180; # 3min

    my $session_id = POE::Component::Client::TCP->new(
        Alias   => $alias_channelserver,
        RemoteAddress => $connect_addr,
        RemotePort    => $connect_port,

        SessionParams => [ options => { debug => $debug, trace => 0, default => 1 } ],

        Started => sub {
            my ($kernel, $heap, $session, $input) = @_[KERNEL, HEAP, SESSION, ARG0];

            # store alias & consumer
            $heap->{alias} = $alias_channelserver;
            $heap->{alias_customer} = $alias_customer;

            # signal handlers
            $kernel->sig(DIE => 'sig_DIE');
        },

        Connected => sub {
            my ($kernel, $heap, $session, $input) = @_[KERNEL, HEAP, SESSION, ARG0];
            my $sid = $session->ID;

            # inform interpreter on new client connection
            $kernel->call($alias_interpreter, 'client_connected',
                          $alias_channelserver, $sap, $sid, $key_id,
                          $alias_customer, $opaque_sdata);

            c_log("connected to channel server at " . $sap);

            # setup PING tick
            $kernel->alarm(tick_PING => time() + $ping_interval);
        },

        Disconnected => sub {
            my ($kernel, $heap, $session, $input) = @_[KERNEL, HEAP, SESSION, ARG0];
            my $sid = $session->ID;

            # clear PING tick
            $kernel->alarm('tick_PING');

            # inform interpreter on lost client connection
            c_log("disconnected from channel server at " . $sap);
            $kernel->post($alias_interpreter, 'client_disconnected', $alias_channelserver, $sap, $sid);
        },

        ServerInput => sub {
            my ($kernel, $heap, $session, $input) = @_[KERNEL, HEAP, SESSION, ARG0];
            my $sid = $session->ID;

            # process server response
            dc_poe_client_process_input($sid, $heap, $input);
        },

        InlineStates  => {
            send => sub {
                my ($kernel, $heap, $session, $output) = @_[KERNEL, HEAP, SESSION, ARG0];

                # send received data to client
                $heap->{server}->put($output);
            },

            disconnect => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];

                # TODO: XXX: disconnect: send a QUIT message to the channel server

                # shutdown
                $kernel->yield('shutdown');
            },

            tick_PING => sub {
                my ($kernel, $heap, $session) = @_[KERNEL, HEAP, SESSION];
                my $sid = $session->ID;

                # make interpreter execute a PING request to $recipient
                $kernel->post($alias_interpreter, 'cmd_PING', $sid, []);

                # setup PING tick
                $kernel->alarm(tick_PING => time() + $ping_interval);
            },

            sig_DIE => sub {
                # $sig is 'DIE', $exception is the exception hash
                my($kernel, $heap, $session, $sig, $exception ) = @_[KERNEL, HEAP, SESSION, ARG0, ARG1];

                # show exception
                my $msg = "died in $exception->{event}\n$exception->{error_str}";
                dc_log_err($msg, 'SIGDIE: Exception: ChannelServer', $alias_channelserver);

                # mark exception as handled
                $poe_kernel->sig_handled();

                # do not shutdown for now on DIE signals. this allows us to see the exception
                # inside the client in the log notebook page
                #$kernel->yield('shutdown');
            },
        },
    );

    return ($session_id, $alias_channelserver);
}

sub dc_poe_client_initialize(;$)
{
    # configure interpreter
    $alias_interpreter = shift // 'Client-Interpreter';

    # initialize this module
    dc_log_dbg("initializing DarkChannel::Node::Client::ChannelServer");

    return 1;
}

1;
