#
# Dark Channel Client Response Library
#
# Copyright (C) 2015 by DataCore GmbH
#     Amir Guindehi <amir@datacore.ch>
#

package DarkChannel::Proto::Client::Response;

use warnings;
use strict;

use Carp;
use Data::Dumper;

use DarkChannel::Crypt::Base;

use DarkChannel::Utils::Log;
use DarkChannel::Utils::Block;
use DarkChannel::Utils::SessionStorage;

use DarkChannel::Proto::V1;

use DarkChannel::Node::Client::Conf;

use POE;

use Exporter;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

our $VERSION = 1.00;
our @ISA = qw( Exporter );
our @EXPORT_OK = qw();
our @EXPORT = qw( dc_client_response_initialize
                  dc_client_response_process );

# consumer session alias
my $CONSUMER = '';

#
# import public key material after inspecting it and verify the signature
#
# returns: $import
#
sub dc_client_response_import($$)
{
    my $sid = shift;
    my $pubkey = shift // confess('no public key!');

    my $alias = dc_session_data_get($sid, 'alias');
    my $key_id = dc_session_data_get($sid, 'key_id');

    # XXX: IDEA: implement option to disallow unknown server connections effectively
    #            disallowing any new server (and so with a unknown public key) from being used
    dc_log_warn("unknown server public key received, eventually a new and unknown server?", $alias);
    dc_log_info("importing new server's public key and re-validating received response", $alias);

    # inspect pubkey and remember keyid
    my $inspect = crypt_base_key_inspect($pubkey); # inspect the received armored pubkey
    my @keyids = keys %{ $inspect };
    if ($#keyids gt 0) {
        dc_log_warn("received response containing more than 1 key!", $alias);
        return 0;
    }
    unless ($inspect->{$keyids[0]}->{key_type} eq 'pub') {
        dc_log_warn("received response containing a non 'pub' key!", $alias);
        return 0;
    }
    my $pubkey_id = $keyids[0];

    # XXX: TODO: FIX: recognize impersonating servers by remembering IP/KEY pairs and inform user
    # XXX: TODO: FIX: recognize new server by remembering IP/KEY pairs and inform user

    # import the armored key if it's a single public key
    my $import = crypt_base_key_import($pubkey);
    my $keyid = 0;

    if ($import) {
        my @keyids = keys %{ $import };
        $keyid = $keyids[0];
    }

    return $keyid;
}

#
# detach last block, assume it's a signature and verify the signature
#
# returns: ($signature_state, $data)
#
#          signature_state: ['valid/known', 'valid/unknown', 'invalid']
#          data:            signed data
#
sub dc_client_response_verify($$)
{
    my $sid = shift;
    my $response = shift;

    my $prefix_response = 'Response';
    my $prefix_sig = 'Signature';

    my $alias = dc_session_data_get($sid, 'alias');
    my $key_id = dc_session_data_get($sid, 'key_id');

    # log response if configured
    dc_client_response_log($sid, $response, $prefix_response)  if ($CONF->{log}->{log_dbg_response});

    # split signature from blob (the last block needs to be the signature, all responses are signed)
    my ($data, $signature) = dc_block_split_last($CONF, $response);

    # check signature and return 0 on failure
    my $sig = crypt_base_data_verify($signature, $response, $key_id);
    my $sig_state = $sig->{state} // 'invalid';
    my $sig_keyid = $sig->{key_id} // 0;

    # log verify failure
    dc_client_response_log($sid, Dumper($sig), $prefix_sig)
        if ($CONF->{log}->{log_dbg_verify} || not($sig_state eq 'valid/known'));

    return ($sig_keyid, $sig_state, $data);
}

#
# decrypt response and verify the signature
#
# returns: ($signature_state, $data)
#
#          signature_state: ['valid/known', 'valid/unknown', 'invalid']
#          data:            decrypted data
#
sub dc_client_response_decrypt_and_verify($$)
{
    my $sid = shift;
    my $response = shift // '';

    my $alias = dc_session_data_get($sid, 'alias');
    my $key_id = dc_session_data_get($sid, 'key_id');
    my $prefix_content = 'Response';
    my $prefix_transport = 'Response Transport';

    # log response if configured
    dc_client_response_log($sid, $response, $prefix_transport)  if ($CONF->{log}->{log_dbg_response_transport});

    # decrypt and check signature
    my $decrypted = crypt_base_data_decrypt($response, $key_id);

    # inspect result
    my $signature_state = $decrypted->{signature}->{state} // 'invalid';
    my $signature_keyid = $decrypted->{signature}->{key_id} // 0;

    my ($nickname_state, $nickname_keyid) = ('invalid', 0);
    $nickname_state = $decrypted->{nickname}->{state} // 'invalid' if ($decrypted->{nickname});
    $nickname_keyid = $decrypted->{nickname}->{key_id} // 0 if ($decrypted->{nickname});

    # log response if configured
    dc_client_response_log($sid, $decrypted->{data}, $prefix_content)  if ($CONF->{log}->{log_dbg_response});

    # remove message terminator
    $decrypted->{data} = $1 if ($decrypted->{data} =~ /^(.*)\.\n?$/s);

    # log result
    if ($CONF->{log}->{log_dbg_verify}) {
        if ($signature_state eq 'valid/known') {
            dc_log_dbg('encrypted response has been successfully decrypted and verified (signer_keyid='
                       . $signature_keyid . ')', $alias . ': Signature');
        }
        elsif ($signature_state eq 'valid/unknown') {
            dc_log_dbg('encrypted response has been successfully decrypted, but signer is unknown to us!',
                       $alias . ': Signature');
        }
        else {
            dc_log_dbg('response could not be decrypted!',  $alias . ': Signature');
        }
    }

    return ($decrypted->{data}, $signature_keyid, $signature_state, $nickname_keyid, $nickname_state);
}


sub dc_client_response_WELCOME($$$$)
{
    my $sid = shift;
    my $cmd = shift;
    my $arg = shift;
    my $pubkey = shift;

    my $alias = dc_session_data_get($sid, 'alias');

    # verify protocol specification: command arguments
    unless ($arg =~ /^$CONF->{product_name}: Channel Server$/)
    {
        dc_log_warn("received 'WELCOME' response with invalid arguments, ignoring response!", $alias);
        return 0;
    }

    # inspect pubkey and remember keyid
    my $inspect = crypt_base_key_inspect($pubkey); # inspect the received armored pubkey
    my @keyids = keys %{ $inspect };
    if ($#keyids gt 0) {
        dc_log_warn("received 'WELCOME' response containing more than 1 key!", $alias);
        return 0;
    }
    if ($inspect->{$keyids[0]}->{key_type} ne 'pub') {
        dc_log_warn("received 'WELCOME' response containing a non 'pub' key!", $alias);
        return 0;
    }
    my $pubkey_id = $keyids[0];

    # XXX: TODO: check received keyid and compare it to expected keyid
    # XXX: TODO: alert user of channelserver keyid change: POSSIBLE MAN IN THE MIDDLE ATTACK!
    # XXX: TODO: sign received channelserver keyid if accepted by user

    # sign new channel server key ($pubkey_id) with client key ($key_id), mark it as trusted introducer
    my $key_id = dc_session_data_get($sid, 'key_id');
    my $sap = dc_session_data_get($sid, 'sap');
    my $domain = (split(/:/, $sap))[0];
    crypt_base_key_sign($key_id, $pubkey_id, 'local-trusted', $domain);

    # consume WELCOME
    return dc_client_response_consume($sid, $cmd, $arg, $pubkey, $pubkey_id);
}

sub dc_client_response_HELLO($$$$)
{
    my $sid = shift;
    my $cmd = shift;
    my $arg = shift;
    my $blob = shift;

    my $alias = dc_session_data_get($sid, 'alias');

    # verify protocol specification: blob has to be empty
    return 0 if ($blob);

    # verify protocol specification: command arguments
    my $protocol_version;
    if ($arg =~ /^$CONF->{product_name}: Channel Server: (v[0-9]+): ([^\s]+)$/) {
        $protocol_version = $1;
    }
    else {
        dc_log_warn("received 'HELLO' response with invalid arguments, ignoring response!", $alias);
        return 0;
    }
    # verify protocol specification: check protocol version
    unless(dc_proto_version($protocol_version))
    {
        dc_log_warn("received 'HELLO' response requesting protocol version '" . $protocol_version .
                    "' but we do not support that version, ignoring response!", $alias);
        return 0;
    }

    # consume HELLO
    return dc_client_response_consume($sid, $cmd, $arg);
}

sub dc_client_response_JOIN($$$$)
{
    my $sid = shift;
    my $cmd = shift;
    my $arg = shift;
    my $blob = shift;

    my $alias = dc_session_data_get($sid, 'alias');

    # verify protocol specification: command arguments
    unless($arg =~ /^#[a-zA-Z-_]+$/)
    {
        dc_log_warn("received 'JOIN' response with invalid arguments, ignoring response!", $alias);
        return 0;
    }

    # inspect pubkey if available
    if ($blob) {
        my $inspect = crypt_base_key_inspect($blob); # inspect the received armored pubkey
        my @keyids = keys %{ $inspect };
        for my $keyid (@keyids) {
            next if ($inspect->{$keyid}->{key_type} eq 'pub');
            dc_log_warn("received 'JOIN' response containing a non 'pub' key!", $alias);
            return 0;
        }
    }

    # consume JOIN
    return dc_client_response_consume($sid, $cmd, $arg, $blob);
}

sub dc_client_response_PART($$$$)
{
    my $sid = shift;
    my $cmd = shift;
    my $arg = shift;
    my $blob = shift;

    my $alias = dc_session_data_get($sid, 'alias');

    # verify protocol specification: command arguments
    unless ($arg =~ /^#[a-zA-Z-_]+ 0x[0-9A-F]+$/)
    {
        dc_log_warn("received 'PART' response with invalid arguments, ignoring response!", $alias);
        return 0;
    }

    # consume PART
    return dc_client_response_consume($sid, $cmd, $arg);
}

sub dc_client_response_RELAY($$$$)
{
    my $sid = shift;
    my $cmd = shift;
    my $arg = shift;
    my $blob = shift;

    my $alias = dc_session_data_get($sid, 'alias');
    my $key_id = dc_session_data_get($sid, 'key_id');

    # verify protocol specification: command arguments
    unless (($arg =~ /^#[a-zA-Z-_]+$/) || ($arg =~ /^0x[0-9A-Z]+$/))
    {
        dc_log_warn("received 'RELAY' response with invalid arguments, ignoring response!", $alias);
        return 0;
    }

    # inspect encrypted data if available
    if ($blob) {
        # consume RELAY
        my $success = dc_client_response_consume($sid, $cmd, $arg);

        # parse encrypted blob and return
        my $encrypted = 1;
        return dc_client_response_process($sid, $blob, $encrypted);
    }

    # no encrypted content
    dc_log_warn("received 'RELAY' response containing no encrypted data!", $alias);
    return 0;
}

sub dc_client_response_MESSAGE($$$$$)
{
    my $sid = shift;
    my $cmd = shift;
    my $arg = shift;
    my $message = shift;
    my $signature_keyid = shift;
    my $nickname_keyid = shift;

    my $alias = dc_session_data_get($sid, 'alias');
    my $key_id = dc_session_data_get($sid, 'key_id');

    # verify protocol specification: command arguments
    unless (($arg =~ /^#[a-zA-Z-_]+$/) || ($arg =~ /^0x[0-9A-Z]+$/))
    {
        dc_log_warn("received 'MESSAGE' response with invalid arguments, ignoring response!", $alias);
        return 0;
    }

    # verify protocol specification: need message blob
    unless($message) {
        dc_log_warn("received 'MESSAGE' response without message, ignoring response!", $alias);
        return 0;
    }

    # consume MESSAGE
    return dc_client_response_consume($sid, $cmd, $arg, $message, $signature_keyid, $nickname_keyid);
}

sub dc_client_response_NICK($$$$)
{
    my $sid = shift;
    my $cmd = shift;
    my $arg = shift;
    my $blob = shift;
    my $signature_keyid = shift;

    my $alias = dc_session_data_get($sid, 'alias');
    my $key_id = dc_session_data_get($sid, 'key_id');

    # verify protocol specification: command arguments
    unless ($arg =~ /^(#[a-zA-Z-_]+)$/) {
        dc_log_warn("received 'NICK' response with invalid arguments, ignoring response!", $alias);
        return 0;
    }

    # verify protocol specification: blob
    unless ($blob) {
        dc_log_warn("received 'NICK' response containing no public key material, ignoring response!", $alias);
        return 0;
    }

    # inspect pubkey if available
    my $inspect = crypt_base_key_inspect($blob); # inspect the received armored pubkey
    my @keyids = keys %{ $inspect };
    if ($#keyids gt 0) {
        dc_log_warn("received 'NICK response containing more than 1 key!", $alias);
        return 0;
    }
    if ($inspect->{$keyids[0]}->{key_type} ne 'pub') {
        dc_log_warn("received 'NICK response containing a non 'pub' key!", $alias);
        return 0;
    }
    my $nick_keyid = $keyids[0];
    my $nick_fq = $inspect->{$nick_keyid}->{uid_email};
    unless ($nick_keyid && $nick_fq) {
        dc_log_warn("received 'NICK' response with invalid public key material, ignoring response!", $alias);
        return 0;
    }

    # consume NICK
    return dc_client_response_consume($sid, $cmd, $arg, $blob, $nick_keyid, $nick_fq, $signature_keyid);
}

sub dc_client_response_PONG($$$$)
{
    my $sid = shift;
    my $cmd = shift;
    my $arg = shift;
    my $blob = shift;

    my $alias = dc_session_data_get($sid, 'alias');

    # verify protocol specification: command arguments
    unless($arg =~ /^\d+$/)
    {
        dc_log_warn("received 'PONG' response with invalid arguments, ignoring response!", $alias);
        return 0;
    }

    # consume PONG
    return dc_client_response_consume($sid, $cmd, $arg);
}

sub dc_client_response_LIST($$$$)
{
    my $sid = shift;
    my $cmd = shift;
    my $arg = shift;
    my $blob = shift;

    my $alias = dc_session_data_get($sid, 'alias');

    # verify protocol specification: no command arguments
    if($arg)
    {
        dc_log_warn("received 'LIST' response with invalid arguments, ignoring response!", $alias);
        return 0;
    }

    # consume LIST
    return dc_client_response_consume($sid, $cmd, $arg, $blob);
}

sub dc_client_response_REGISTER($$$$)
{
    my $sid = shift;
    my $cmd = shift;
    my $arg = shift;
    my $blob = shift;

    my $alias = dc_session_data_get($sid, 'alias');

    # verify protocol specification: command arguments
    my ($role, $name);
    if($arg =~ /^([A-Z]+) ([a-zA-Z-_]+)$/) {
        if ($1 ne 'NICKNAME') {
            dc_log_warn("received not supported 'REGISTER $1' response, ignoring response!", $alias);
            return 0;
        }
        $role = $1;
        $name = $2;
    }
    else {
        dc_log_warn("received 'REGISTER' response with invalid argument, ignoring response!", $alias);
        return 0;
    }

    # inspect pubkey if available
    my $pubkey_id;
    if ($blob) {
        my $inspect = crypt_base_key_inspect($blob); # inspect the received armored pubkey
        my @keyids = keys %{ $inspect };
        if ($#keyids gt 0) {
            dc_log_warn("received 'REGISTER " . $role . "' response containing more than 1 key!", $alias);
            return 0;
        }
        if ($inspect->{$keyids[0]}->{key_type} ne 'pub') {
            dc_log_warn("received 'REGISTER " . $role . "' response containing a non 'pub' key!", $alias);
            return 0;
        }
        $pubkey_id = $keyids[0];
    }

    # consume REGISTER
    return dc_client_response_consume($sid, $cmd, $arg, $role, $name, $blob, $pubkey_id);
}

#
# send consume event to consumer
#
# dc_client_response_consume($sid, $cmd, $arg, ...)
#

sub dc_client_response_consume($$$;@)
{
    my ($sid, $cmd, $arg) = (shift, shift, shift);
    my $params = \@_;
    my $event = 'consume_' . $cmd;

    $poe_kernel->post($CONSUMER, $event, $sid, $arg, $params)
        || confess("failed to send event '" . $event . "' to consumer '" . $CONSUMER . "'!");
    return 1;
}

sub dc_client_response_log($$$)
{
    my ($sid, $message, $prefix) = @_;
    my $alias = dc_session_data_get($sid, 'alias');
    my $alias_customer = dc_session_data_get($sid, 'alias_customer');
    my $sap = dc_session_data_get($sid, 'sap');

    confess('no $alias_customer in dc_client_response_log()!') unless($alias_customer);

    # inform customer
    $poe_kernel->post($alias_customer, 'darkchannel_LOG_PROTO', $sid, $sap, $message, $prefix);

}

sub dc_client_response_process($$;$)
{
    my ($sid, $response, $encrypted) = @_;

    my $transport_encryption = $encrypted // $CONF->{settings}->{transport_encryption} // 1;
    my ($data, $sig_keyid, $sig_state, $nick_keyid, $nick_state) = ('', 0, 'invalid', 0, 'invalid');

    # WELCOME is the only command allowed to be unencrypted and unsigned with transport encryption
    if ($response !~ /^WELCOME /) {
        if ($transport_encryption) {
            # this clients signature key has to be known and decryption & signature verification has to succeed
            ($data, $sig_keyid, $sig_state, $nick_keyid, $nick_state)
                = dc_client_response_decrypt_and_verify($sid, $response);
        }
        else {
            # this clients signature key has to be known and signature verification has to succeed
            ($sig_keyid, $sig_state, $data) = dc_client_response_verify($sid, $response);
        }
    }
    else {
        # WELCOME message is not encrypted nor signed
        $data = $response;
        $sig_keyid = 0;
        $sig_state = 'valid/unknown';

        # log response if configured
        dc_client_response_log($sid, $response, 'Response')  if ($CONF->{log}->{log_dbg_response});
    }

    # parse received response
    my @lines = split(/\n/, $data);
    my $alias = dc_session_data_get($sid, 'alias');
    my ($cmd, $arg, $blob) = ('', '', '');

    if ($lines[0] && (($lines[0] =~ /^([A-Z]+)$/) || ($lines[0] =~ /^([A-Z]+) (.*)$/))) {
        ($cmd, $arg) = ($1, $2);
        shift(@lines);
        $blob = join("\n", @lines);
    }
    else {
        dc_log_warn("response command is not well formed [A-Z]!", $alias);
        return 0;
    }

    # WELCOME & HELLO are the only command allowed to be unsigned and unencrypted,
    # for all others verify has to succeed
    unless($sig_state eq 'valid/known') {
        # do not fail if WELCOME and HELLO messages are not verified
        unless(($cmd eq 'WELCOME') && ($sig_state eq 'valid/unknown')) {
            dc_log_warn("received response with invalid signature (sig_keyid=" . $sig_keyid
                        . ", sig_state=" . $sig_state . "), ignoring response!", $alias);
            return 0;
        }
        # if this is a server connected for the first time, it will sign with an unknown public key
        # and so signature verification will fail. handle this gracefully by importing the
        # public key prior to re-validate the response
        #
        # sig_state stays 'valid/unknown' so that dc_client_response_WELCOME() knows
        $sig_keyid = dc_client_response_import($sid, $data);
    }

    # clear nick keyid if not valid/known
    $nick_keyid = 0 if ($nick_state ne 'valid/known');

    # call CLIENT RESPONSE function (CONSUMER)
    my $response_func_name = 'dc_client_response_' . $cmd;
    dc_client_response_log($sid, "'" . $response_func_name . "()'", 'Response: Call Consumer')
        if ($CONF->{log}->{log_dbg_transition});

    # check if response function exists and call it
    if (DarkChannel::Proto::Client::Response->can($response_func_name)) {
        # XXX: TODO: dont confess() but react accordingly on failed response intergrity check
        my $response_func = \&$response_func_name;
        my $success = $response_func->($sid, $cmd, $arg, $blob, $sig_keyid, $nick_keyid);
        unless($success) {
            dc_log_warn("response integrity check failed in '$response_func_name'!");
            return 0;
        }
        return 1;
    }
    dc_log_warn("response function'" . $response_func_name . "()' does not exist!");
    return 0;
}

sub dc_client_response_initialize(;$)
{
    $CONSUMER = shift // 'Client-Interpreter';

    # initialize logging
    dc_log_dbg("initializing DarkChannel::Proto::Client::Response (consumer=" . $CONSUMER . ")");

    return 1;
}

1;
