#
# Dark Channel Server Response Library
#
# Copyright (C) 2015 by DataCore GmbH
#     Amir Guindehi <amir@datacore.ch>
#

package DarkChannel::Proto::Server::Response;

use warnings;
use strict;

use Carp;
use Data::Dumper;

use DarkChannel::Crypt::Base;

use DarkChannel::Utils::Log;
use DarkChannel::Utils::SessionStorage;

use DarkChannel::Node::ChannelServer::Conf;

use POE;

use Exporter;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

our $VERSION = 1.00;
our @ISA = qw( Exporter );
our @EXPORT_OK = qw();
our @EXPORT = qw( dc_server_response_initialize
                  dc_server_response_send );

sub dc_server_response($$;$)
{
    my $sid = shift;
    my $response = shift;
    my $plain = shift // 0;

    my $transport_encryption = $CONF->{settings}->{transport_encryption} // 1;

    my $prefix_response = 'Response';
    my $alias = dc_session_data_get($sid, 'alias');
    my $key_id = dc_session_data_get($sid, 'channelserver', 'key_id');
    my $recipient_key_id = dc_session_data_get($sid, 'client', 'key_id');
    my $response_transport;

    if (not $plain) {
        if ($transport_encryption) {
            # add response end marker
            $response .= "\n.\n";

            # encrypt and sign response
            my $encrypted = crypt_base_data_encrypt($response, $key_id, $recipient_key_id);

            # log error if failed to sign
            unless ($encrypted) {
                my $err = "failed to to encrypt and sign response data, ignoring response!";
                dc_log_crit($err, $alias . ': ' . $prefix_response);
                return 0;
            }

            #  store encrytped and signed response
            $response_transport = $encrypted;
        }
        else {
            # no transport encryption: sign response if requested
            my $signature = crypt_base_data_sign($response, $key_id);

            unless ($signature) {
                my $err = "failed to add signature to response data, ignoring response!";
                dc_log_crit($err, $alias . ': ' . $prefix_response);
                return 0;
            }

            # combine response and signature
            my $prefix = ($response =~ m{\n}) ? "\n-" : '';
            $response_transport .= $prefix . "\n" . $signature;
        }
    }
    else {
        $response_transport = $response;
    }

    # response end marker
    $response_transport .= "\n.";

    return [ $response_transport, $response ];
}

sub dc_server_response_WELCOME($$)
{
    my $sid = shift;
    my $args = shift;

    my $key_id = dc_session_data_get($sid, 'channelserver', 'key_id');
    my $pubkey = crypt_base_key_export_pub($key_id);
    my @banner = (
        $CONF->{product_name},
        $CONF->{service_name},
    );
    my $response = 'WELCOME' . ' ' . join(': ', @banner) . "\n" . $pubkey;

    return dc_server_response($sid, $response, 1);
}

sub dc_server_response_HELLO($$)
{
    my $sid = shift;
    my $args = shift;

    my $protocol_version = dc_session_data_get($sid, 'client', 'protocol_version') // confess("no protocol version!");
    my @banner = (
        $CONF->{product_name},
        $CONF->{service_name},
        $protocol_version,
        $CONF->{server_type},
    );
    my $response = 'HELLO' . ' ' . join(': ', @banner);

    return dc_server_response($sid, $response);
}

sub dc_server_response_JOIN($$)
{
    my $sid = shift;
    my $args = shift;

    my ($channel, $pubkey) = @{$args};
    my $response = 'JOIN' . ' ' . $channel . (($pubkey) ? "\n" . $pubkey : '');

    return dc_server_response($sid, $response);
}

sub dc_server_response_PART($$)
{
    my $sid = shift;
    my $args = shift;

    my ($channel, $keyid) = @{$args};
    my $response = 'PART' . ' ' . $channel . ' '. $keyid;

    return dc_server_response($sid, $response);
}

sub dc_server_response_RELAY($$)
{
    my $sid = shift;
    my $args = shift;

    my ($channel, $encrypted) = @{$args};
    my $response = 'RELAY' . ' ' . $channel . "\n" . $encrypted;

    return dc_server_response($sid, $response);
}

sub dc_server_response_PONG($$)
{
    my $sid = shift;
    my $args = shift;

    my ($timestamp) = @{$args};
    my $response = 'PONG' . ' ' . $timestamp;

    return dc_server_response($sid, $response);
}

sub dc_server_response_LIST($$)
{
    my $sid = shift;
    my $args = shift;
    my ($list) = @{$args};

    my $response = 'LIST';
    foreach my $channel (keys %{$list}) {
        my $topic = $list->{$channel}->{topic} // '';
        $response .= "\n";
        $response .= $channel . ($topic ? ' ' . $topic : '');
    }

    return dc_server_response($sid, $response);
}

sub dc_server_response_REGISTER($$)
{
    my $sid = shift;
    my $args = shift;

    my ($role, $name, $pubkey) = @{$args};
    my $response = 'REGISTER' . ' ' . uc($role) . ' ' . $name . (($pubkey) ? "\n" . $pubkey : '');

    return dc_server_response($sid, $response);
}

sub dc_server_response_send($$;$)
{
    my ($sid, $response, $response_args) = @_;

    return if ($response eq 'NOP');

    # call SERVER RESPONSE function (PRODUCER)
    my $alias = dc_session_data_get($sid, 'alias');
    my $response_func_name = 'dc_server_response_' . $response;
    dc_log_dbg("'" . $response_func_name . "()'", $alias . ': Response: Call Producer')
        if ($CONF->{log}->{log_dbg_transition});

    if ($response =~ /^([A-Z]+)$/) {
        # check if response function exists
        if (DarkChannel::Proto::Server::Response->can($response_func_name)) {
            # XXX: TODO: dont confess() but react accordingly on failed response generation
            my $prefix_content = 'Response';
            my $prefix_transport = 'Response Transport';
            my $response_func = \&$response_func_name;

            # call response function and store response data
            my $resp = $response_func->($sid, $response_args)
                || confess("response generation failed in '$response_func_name'!");
            my ($response_transport, $response_content) = @{ $resp };

            # log response
            dc_log_dbg($response_content, $alias . ': ' . $prefix_content)
                if ($CONF->{log}->{log_dbg_response});
            dc_log_dbg($response_transport, $alias . ': ' . $prefix_transport)
                if ($CONF->{log}->{log_dbg_response_transport});

            # send output response event to client session
            $poe_kernel->post($sid, 'send', $response_transport);
        }
        else {
            confess("response function'" . $response_func_name . "()' does not exist!");
        }
    }
    else {
        confess("response command '" . $response . " is not well formed [A-Z]!");
    }
}

sub dc_server_response_initialize()
{
    # initialize logging
    dc_log_dbg("initializing DarkChannel::Proto::Server::Response");

    return 1;
}

1;
