#
# Dark Channel Log Library
#
# Copyright (C) 2015 by DataCore GmbH
#     Amir Guindehi <amir@datacore.ch>
#

package DarkChannel::Utils::Log;

use warnings;
use strict;

use Carp;
use Data::Dumper;
use POSIX qw(strftime);

use DarkChannel::Utils::Log::STDERR;

use Exporter;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

our $VERSION = 1.00;
our @ISA = qw( Exporter );
our @EXPORT_OK = qw();
our @EXPORT = qw( dc_log_initialize
                  dc_log_shutdown
                  dc_log_tie_stderr
                  dc_log_untie_stderr
                  dc_log_set_terminal
                  dc_log
                  dc_log_info
                  dc_log_warn
                  dc_log_err
                  dc_log_crit
                  dc_log_dbg );

# we need a local copy of $CONF as this module is used by Client
# _and_ ChannelServer and so we can not just use the right module
my $CONF = {};

my $TERM = {
    support     => 0,
    active      => 0,
    detach      => 0,
    text_log    => undef,
    date_format => '',
    buffer      => [],
};

my $LOGFILE;
my $LOGFILENAME;

#
# append formatted arrayref of lines to Curses::UI::TextEditor
#
# dc_log_append_formated(@lines)
#
sub dc_log_append_formated($)
{
    my $linesref = shift // confess("No 'linesref' given!");
    my @lines = @{$linesref};
    my $lines = join("\n", @lines);
    my $text = $TERM->{text_log}->text();
    my $newtext = $text . "\n" . $lines;

    # make sure buffer does not get to big
    my $max_bufsize = $CONF->{settings}->{buffer_size};
    my $len = length($newtext);
    if ($len > $max_bufsize) {
        $newtext = substr($newtext, $len - $max_bufsize);
        # find newline
        while(substr($newtext, 0, 1) ne "\n") {
            $newtext = substr($newtext, 1);
        }
        # remove newline to be sure to be at the beginning of a line
        $newtext = substr($newtext, 1);
    }

    # log to UI::TextEditor
    $TERM->{text_log}->text($newtext);
    $TERM->{text_log}->cursor_to_end();

    # let UI::Notebook (the parent of the page containing the UI::TextEditor) redraw
    $TERM->{text_log}->parent()->parent()->draw();

    # log to logfile if detached and log to terminal and logfile if not
    unless ($TERM->{detach}) {
        print $lines . "\n";
    }
    print $LOGFILE $lines . "\n";
}

#
# generic log function
#
# dc_log($msg, $prefix)
#
sub dc_log($@)
{
    my $msg = shift;
    my $prefix = shift // '';
    my $now = strftime($TERM->{date_format}, localtime);
    my @lines = split(/\n/, $msg);
    my $stamp = "[" . $now . "] " . $CONF->{service_name} . ': ' . ($prefix ? $prefix . ': ' : '');
    my @out = ();

    foreach my $line (@lines)
    {
        my $entry = $stamp . $line;

        # if terminal support is enabled, store messages while terminal
        # is not yet running, otherwise just remember them for output
        if ($TERM->{support}) {
            # use terminal window if active, buffer otherwise
            push(@out, $entry) if ($TERM->{active});
            push(@{$TERM->{buffer}}, $entry) if (not $TERM->{active});
        }
        else {
            push(@out, $entry);
        }
    }

    if (($TERM->{support}) && ($TERM->{active})) {
        dc_log_append_formated(\@out);
    }
    elsif (@out) {
        if ($TERM->{detach}) {
            print $LOGFILE join("\n", @out) . "\n";
        }
        else {
            print join("\n", @out) . "\n";
            print $LOGFILE join("\n", @out) . "\n";
        }
    }
}

#
# activate Curses::UI::TextEditor logging and replays prior messages
#
# dc_log_set_terminal($text_log)
#
sub dc_log_set_terminal($)
{
    # activate UI logging
    $TERM->{text_log} = shift;
    $TERM->{support} = 1;
    $TERM->{active} = 1;

    # replay buffered messages
    my @lines = @{$TERM->{buffer}};
    dc_log_append_formated(\@lines);
    $TERM->{buffer} = [];
}

sub dc_log_crit($@)
{
    my $msg = shift;
    my $prefix = shift // '';
    my $newprefix = 'CRIT' . ($prefix ? ': ' . $prefix : '');
    dc_log($msg, $newprefix) if ($CONF->{log}->{log_crit});
}

sub dc_log_err($@)
{
    my $msg = shift;
    my $prefix = shift // '';
    my $newprefix = 'ERR' . ($prefix ? ': ' . $prefix : '');
    dc_log($msg, $newprefix) if ($CONF->{log}->{log_err});
}

sub dc_log_warn($@)
{
    my $msg = shift;
    my $prefix = shift // '';
    my $newprefix = 'WARN' . ($prefix ? ': ' . $prefix : '');
    dc_log($msg, $newprefix) if ($CONF->{log}->{log_warn});
}

sub dc_log_info($@)
{
    my $msg = shift;
    my $prefix = shift // '';
    my $newprefix = 'INFO' . ($prefix ? ': ' . $prefix : '');
    dc_log($msg, $newprefix) if ($CONF->{log}->{log_info});
}

sub dc_log_dbg($@)
{
    my $msg = shift;
    my $prefix = shift // '';
    my $newprefix = 'DBG' . ($prefix ? ': ' . $prefix : '');
    dc_log($msg, $newprefix) if ($CONF->{log}->{log_dbg});
}

sub dc_log_tie_stderr()
{
    tie(*STDERR, 'DarkChannel::Utils::Log::STDERR' , sub { dc_log_warn(join(' ', @_), 'STDERR:'); });
}

sub dc_log_untie_stderr()
{
    untie(*STDERR);
}

#
# initialize DarkTable::Utils::Log subsystem with or without terminal support
#
# dc_log_initialize($CONF, $term_support)
#

sub dc_log_initialize($$;$$$)
{
    # initialize globals
    $CONF = shift;
    my $storage_dir = shift;
    $TERM->{support} = shift // 0;
    $TERM->{detach} = shift // 1;
    $TERM->{date_format} = shift // '%Y-%m-%d %H:%M';

    # log file
    $LOGFILENAME = $CONF->{log}->{logfile};
    $LOGFILENAME = $storage_dir . '/' . $LOGFILENAME unless ($LOGFILENAME =~ /^\//);

    # create logfile folder if not existing
    my $storage_mode = 0700;
    if ($LOGFILENAME =~ /^(.*)\/[^\/]+$/) {
        my $logdir = $1;
        if (! -d $logdir) {
            File::Path::make_path($logdir, { verbose => 0, mode => $storage_mode })
                or confess("failed to create folder " . $logdir . "!");
        }
    }

    # clear logfile if existing
   unlink($LOGFILENAME) if (-w $LOGFILENAME);

    # open logfile and store file handle
    open($LOGFILE, ">$LOGFILENAME")
        || confess("could not open logfile '" . $LOGFILENAME . "'");

    # tie STDERR to dc_log_dbg
    dc_log_tie_stderr();

    # print first message
    dc_log_dbg("initializing DarkChannel::Utils::Log (storage-dir=" . $storage_dir . ", term_support="
               . $TERM->{support} . ", term_detach=" . $TERM->{detach} . ")");

    return 1;
}

sub dc_log_shutdown(;$)
{
    my $dump_warn = shift // 0;

    dc_log_dbg("POE main loop stopped, all POE state machines have terminated");
    dc_log_dbg("shutdown DarkChannel::Utils::Log");

    # replay buffered messages
    my @lines = @{$TERM->{buffer}};

    if (@lines) {
        if ($TERM->{detach}) {
            print $LOGFILE join("\n", @lines) . "\n";
        }
        else {
            print join("\n", @lines) . "\n";
        }
    }

    # close log file
    close($LOGFILE);


    # dump all warnings if requested
    if ($dump_warn)
    {
        my $log;
        my $count = 0;
        confess("could not open logfile '" . $LOGFILENAME . "'") if (not open($log, '<' . $LOGFILENAME));
        while (my $line = <$log>)
        {
            next if ($line =~ /^\[.+\] [^:]+:/);
            print $line;
            $count++;
        }
        close($log);

        print "No warnings found.\n" if (not $count);
    }
}

1;
