#
# Dark Channel Session Storage Utility Library
#
# Copyright (C) 2015 by DataCore GmbH
#     Amir Guindehi <amir@datacore.ch>
#

package DarkChannel::Utils::SessionStorage;

use warnings;
use strict;

use Carp;
use Data::Dumper;

use DarkChannel::Utils::Log;

use Exporter;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

our $VERSION = 1.00;
our @ISA = qw( Exporter );
our @EXPORT_OK = qw();
our @EXPORT = qw( dc_session_register
                  dc_session_unregister
                  dc_session_data_get
                  dc_session_data_set
                  dc_session_data_delete
                  dc_session_all
                  dc_session_exists );

# client state indexed by $sid
my $STORAGE = {};

#
# register session with given hash
#
sub dc_session_register($$)
{
    my $sid = shift;
    my $sdata = shift;
    my $alias = $sdata->{alias} // 'Unknown';

    dc_log_dbg("session register (sid=" . $sid . ")", $alias . ': Session');
    $STORAGE->{$sid} = $sdata;
}

#
# unregister session
#
sub dc_session_unregister($)
{
    my $sid = shift;
    my $sdata = $STORAGE->{$sid};
    my $alias = $sdata->{alias} // 'Unknown';

    dc_log_dbg("session unregister (sid=" . $sid . ")", $alias . ': Session');
    return delete $STORAGE->{$sid};
}

#
# gets data of session <sid>
#
# $sessions = dc_session_data_get()
# $session = dc_session_data_get($sid)
# $val = dc_session_data_get($sid, $key)
# $val = dc_session_data_get($sid, $key, $key2)
# $val = dc_session_data_get($sid, $key, $key2, $key3)
#
sub dc_session_data_get(;$$$$$$)
{
    my $sid = shift // undef;
    my $key = shift // 0;
    my $key2 = shift // 0;
    my $key3 = shift // 0;
    my $key4 = shift // 0;
    my $key5 = shift // 0;

    return $STORAGE if (not $sid);
    return $STORAGE->{$sid} if (not $key);
    return $STORAGE->{$sid}->{$key} if (not $key2);
    return $STORAGE->{$sid}->{$key}->{$key2} if (not $key3);
    return $STORAGE->{$sid}->{$key}->{$key2}->{$key3} if (not $key4);
    return $STORAGE->{$sid}->{$key}->{$key2}->{$key3}->{$key4} if (not $key5);
    return $STORAGE->{$sid}->{$key}->{$key2}->{$key3}->{$key4}->{$key5};
}

#
# sets data of session <sid>
#
# $sessions = dc_session_data_set()
# $sessions = dc_session_data_set($sessions)
# $newval = dc_session_data_set($session, $newval)
# $newval = dc_session_data_set($session, $key, $newval)
# $newval = dc_session_data_set($session, $key, $key2, $newval)
#
sub dc_session_data_set(;$$$$$$)
{
    my $sid = shift // undef;
    my $key = shift // 0;
    my $key2 = shift // 0;
    my $key3 = shift // 0;
    my $key4 = shift // 0;
    my $key5 = shift // 0;

    return ($STORAGE = {}) if (not $sid);
    return ($STORAGE = $sid) if (not $key);
    return ($STORAGE->{$sid} = $key) if (not $key2);
    return ($STORAGE->{$sid}->{$key} = $key2) if (not $key3);
    return ($STORAGE->{$sid}->{$key}->{$key2} = $key3) if (not $key4);
    return ($STORAGE->{$sid}->{$key}->{$key2}->{$key3} = $key4) if (not $key5);
    return ($STORAGE->{$sid}->{$key}->{$key2}->{$key3}->{$key4} = $key5);
}

#
# delete data of session <sid>
#
# $oldsessions = dc_session_data_delete()
# $oldsession = dc_session_data_delete($sid)
# $oldval = dc_session_data_delete($sid, $key)
# $oldval = dc_session_data_delete($sid, $key, $key2)
# $oldval = dc_session_data_delete($sid, $key, $key2, $key3)
#
sub dc_session_data_delete(;$$$$)
{
    my $sid = shift // undef;
    my $key = shift // 0;
    my $key2 = shift // 0;
    my $key3 = shift // 0;
    my $storage = $STORAGE;

    $STORAGE = {}, return $storage if (not $sid);
    return delete $STORAGE->{$sid} if (not $key);
    return delete $STORAGE->{$sid}->{$key} if (not $key2);
    return delete $STORAGE->{$sid}->{$key}->{$key2} if (not $key3);
    return delete $STORAGE->{$sid}->{$key}->{$key2}->{$key3};
}

#
# returns 1 if session <sid> exists
#
sub dc_session_exists($)
{
    my $sid = shift;
    return (dc_session_data_get($sid) ? 1 : 0)
}

#
# returns an array of all sid's
#
sub dc_session_all()
{
    return keys %{$STORAGE};
}
